unit Unit1;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ExtCtrls, Menus, CoolTrayIcon, ComCtrls, IniFiles,
  PingSend, TlntSend, WinSock, ImgList;

const
  cDefNumTimeouts = 5;
  cDefTimeout = 1000;
  cNumHosts = 4;
  cDefTimer = 2000;
  cIniFileName = 'watchdog.ini';
  cLogFileName = 'watchdog.log';
  cKomFileName = 'comm.log';
  cResetTimeout = 60000; // minuta

type
  TThReset = class(TThread)
  private
    Telnet: TTelnetSend;
    Start: Cardinal;
  public
    Krok: integer;
    constructor Create;
    procedure Execute; override;
    destructor Destroy; override;
  end;

  TPingTh = class(TThread)
  private
    PingSend: TPINGSend;
    ID: Integer; // ID Threadu
    function GetTimeout: Integer;
    procedure SetTimeout(value: Integer);
  public
    Host: string;
    property Timeout: Integer read GetTimeout write SetTimeout;

    constructor Create(const Host: string; const Timeout: Cardinal; fID: integer);
    procedure Execute; override;
    destructor Destroy; override;
  end;

  TPingObj = class(TObject)
  private
    lblPing, lblAvg, lblPL: TLabel;
    Count, PLCount: Cardinal;
    Avg: Real;
    ID: integer;
    PLInRow: Integer;
  public
    Addr, Disp: string;
    CanReset: Boolean;
    procedure AddPing(const time: Integer);
    constructor Create(fPing, fAvg, fPL: TLabel; fID: integer);
  end;

  TMainForm = class(TForm)
    CoolTrayIcon1: TCoolTrayIcon;
    PopupMenu1: TPopupMenu;
    Obnovit1: TMenuItem;
    Zav1: TMenuItem;
    StatusBar1: TStatusBar;
    Host1: TLabel;
    Host2: TLabel;
    Host3: TLabel;
    Host4: TLabel;
    P1: TLabel;
    P2: TLabel;
    P3: TLabel;
    P4: TLabel;
    A1: TLabel;
    A2: TLabel;
    A3: TLabel;
    A4: TLabel;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    PL1: TLabel;
    PL2: TLabel;
    PL3: TLabel;
    PL4: TLabel;
    ImageList1: TImageList;
    MainMenu1: TMainMenu;
    Soubor1: TMenuItem;
    Nastaven1: TMenuItem;
    Oprogramu1: TMenuItem;
    N1: TMenuItem;
    Hide1: TMenuItem;
    Ukonit1: TMenuItem;
    Bevel1: TBevel;
    Image1: TImage;
    procedure Zav1Click(Sender: TObject);
    procedure Obnovit1Click(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure Nastaven1Click(Sender: TObject);
    procedure Oprogramu1Click(Sender: TObject);
  private
    CanClose: Boolean;
  public
    procedure Log(str: string);
    procedure WMUser(var msg: TMessage); message wm_User;
    procedure ResetError;
    procedure ResetStart;
    procedure ResetOK;
    procedure ResetProblem;
  end;

function CurDir: string;
function WSGetHostByName(const AHostName: string): string;

var
  MainForm: TMainForm;
  hEvTerminate, hEvReset: THandle;
  ThReset: TThReset;
  cTimer: integer;
  cNumTimeouts: Integer;
  ResetHost: string;
  Challenges, Answers: TStrings;
  PingObjs: array[0..cNumHosts-1] of TPingObj;
  PingThs: array[0..cNumHosts-1] of TPingTh;

implementation

uses fAbout, fSetting;

{$R *.dfm}

function CurDir: string;
var
  Dir: string;
begin
  GetDir(0, Dir);
  if Dir[Length(Dir)] <> '\' then Dir := Dir + '\';
  Result := Dir;
end;

function WSGetHostByName(const AHostName: string): string;
var
  pa: PChar;
  Host: PHostEnt;
begin
  Host := GetHostByName(PChar(AHostName));
  if Host = nil then Result := AHostName
  else
  begin
    pa := Host^.h_addr_list^;
    result := IntToStr(Ord(pa[0])) + '.' + IntToStr(Ord(pa[1])) + '.'
      + IntToStr(Ord(pa[2])) + '.' + IntToStr(Ord(pa[3]));
  end;
end;

procedure TMainForm.Zav1Click(Sender: TObject);
begin
  CanClose := True;
  Close;
end;

procedure TMainForm.Obnovit1Click(Sender: TObject);
begin
  if MainForm.Visible then
    CoolTrayIcon1.HideMainForm
  else
    CoolTrayIcon1.ShowMainForm;
end;

{ TPingTh }

constructor TPingTh.Create(const Host: string; const Timeout: Cardinal; fID: integer);
begin
  inherited Create(True);

  FreeOnTerminate := False;
  ID := fID;
  self.Host := Host;
  PingSend := TPingSend.Create;
  PingSend.Timeout := TimeOut;
  self.Timeout := timeout;
end;

destructor TPingTh.Destroy;
begin
  PingSend.Free;

  inherited;
end;

procedure TPingTh.Execute;
begin
  while WaitForSingleObject(hEvTerminate, cTimer) <> WAIT_OBJECT_0 do
  begin
    try
      if PingSend.Ping(Host) then
        SendMessage(MainForm.Handle, WM_USER, ID, PingSend.PingTime)
      else
        SendMessage(MainForm.Handle, WM_USER, ID, -1);
    except
      SendMessage(MainForm.Handle, WM_USER, ID, -1);
    end;
  end;
end;

procedure TMainForm.FormCreate(Sender: TObject);
var
  i: integer;
  IniFile: TIniFile;
  hostname: string;
  timeout: integer;
  ChCnt: integer;
begin
  CanClose := False;

  PingObjs[0] := TPingObj.Create(P1, A1, PL1, 0);
  PingObjs[1] := TPingObj.Create(P2, A2, PL2, 1);
  PingObjs[2] := TPingObj.Create(P3, A3, PL3, 2);
  PingObjs[3] := TPingObj.Create(P4, A4, PL4, 3);

  hEvTerminate := CreateEvent(nil, True, False, nil);
  hEvReset := CreateEvent(nil, False, True, nil);
  ThReset := nil;

  IniFile := TIniFile.Create(CurDir + cIniFileName);

  cNumTimeouts := IniFile.ReadInteger('Constants', 'MaxTimeouts', cDefNumTimeouts);
  IniFile.WriteInteger('Constants', 'MaxTimeouts', cNumTimeouts);

  cTimer := IniFile.ReadInteger('Constants', 'PingInterval', cDefTimer);
  IniFile.WriteInteger('Constants', 'PingInterval', cTimer);

  ResetHost := IniFile.ReadString('ResetHost', 'Hostname', '192.168.168.1');
  IniFile.WriteString('ResetHost', 'Hostname', ResetHost);

  Challenges := TStringList.Create;
  Answers := TStringList.Create;
  ChCnt := IniFile.ReadInteger('ResetHost', 'ChallengesCount', 0);
  if ChCnt = 0 then
  begin
    ChCnt := 3;
    IniFile.WriteInteger('ResetHost', 'ChallengesCount', 3);
    IniFile.WriteString('ResetHost', 'Challenge1', 'password:');
    IniFile.WriteString('ResetHost', 'Answer1', 'password');
    IniFile.WriteString('ResetHost', 'Challenge2', 'Command>');
    IniFile.WriteString('ResetHost', 'Answer2', 'reset system');
    IniFile.WriteString('ResetHost', 'Challenge3', '(y/n)?');
    IniFile.WriteString('ResetHost', 'Answer3', 'y');
  end;

  for i := 1 to ChCnt do
  begin
    Challenges.Add(IniFile.ReadString('ResetHost', 'Challenge'+IntToStr(i),''));
    Answers.Add(IniFile.ReadString('ResetHost', 'Answer'+IntToStr(i),''));
  end;

  Host1.Caption := IniFile.ReadString('Host1', 'DisplayName', 'www.tomorrows.cz');
  Host2.Caption := IniFile.ReadString('Host2', 'DisplayName', 'www.cpx.cz');
  Host3.Caption := IniFile.ReadString('Host3', 'DisplayName', 'www.cesnet.cz');
  Host4.Caption := IniFile.ReadString('Host4', 'DisplayName', 'AP');

  IniFile.WriteString('Host1', 'DisplayName', Host1.Caption);
  IniFile.WriteString('Host2', 'DisplayName', Host2.Caption);
  IniFile.WriteString('Host3', 'DisplayName', Host3.Caption);
  IniFile.WriteString('Host4', 'DisplayName', Host4.Caption);

  PingObjs[0].CanReset := IniFile.ReadBool('Host1', 'TimeoutDoReset', True);
  PingObjs[1].CanReset := IniFile.ReadBool('Host2', 'TimeoutDoReset', True);
  PingObjs[2].CanReset := IniFile.ReadBool('Host3', 'TimeoutDoReset', True);
  PingObjs[3].CanReset := IniFile.ReadBool('Host4', 'TimeoutDoReset', False);

  for i := 0 to cNumHosts - 1 do
  begin
    hostname := IniFile.ReadString('Host' + IntToStr(i+1), 'Hostname', '');
    timeout := IniFile.ReadInteger('Host' + IntToStr(i+1), 'Timeout', cDefTimeout);

    if Hostname = '' then
    begin
      case i+1 of
        1: Hostname := 'www.tomorrows.cz';
        2: Hostname := 'www.cpx.cz';
        3: Hostname := 'www.cesnet.cz';
        4: Hostname := '192.168.168.1';
      end;
    end;

    case i+1 of
      1: PingObjs[i].Disp := Host1.Caption;
      2: PingObjs[i].Disp := Host2.Caption;
      3: PingObjs[i].Disp := Host3.Caption;
      4: PingObjs[i].Disp := Host4.Caption;
    end;

    IniFile.WriteString('Host' + IntToStr(i+1), 'Hostname', hostname);
    IniFile.WriteInteger('Host' + IntToStr(i+1), 'Timeout', timeout);
    IniFile.WriteBool('Host' + IntToStr(i+1), 'TimeoutDoReset', PingObjs[i].CanReset);

    PingObjs[i].Addr := WSGetHostByName(hostname);
    PingThs[i] := TPingTh.Create(WSGetHostByName(hostname), timeout, i);
  end;
  IniFile.Free;

  for i := 0 to cNumHosts - 1 do
    PingThs[i].Resume;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
var
  i: integer;
  x: cardinal;
begin
  for i := 0 to cNumHosts - 1 do
    PingThs[i].Terminate;
  SetEvent(hEvTerminate);
  for i := 0 to cNumHosts - 1 do
  begin
    if WaitForSingleObject(PingThs[i].Handle, PingThs[i].PingSend.Timeout) <> WAIT_OBJECT_0 then
    begin
      GetExitCodeThread(PingThs[i].Handle, x);
      TerminateThread(PingThs[i].Handle, x);
    end;
    PingThs[i].Free;
  end;

  if WaitForSingleObject(hEvReset, 0) <> WAIT_OBJECT_0 then
  begin
    ThReset.Terminate;
    if WaitForSingleObject(ThReset.Handle, 1000) <> WAIT_OBJECT_0 then
    begin
      GetExitCodeThread(ThReset.Handle, x);
      TerminateThread(ThReset.Handle, x);
    end;
    ThReset.Free;
  end;

  CloseHandle(hEvTerminate);
  CloseHandle(hEvReset);
  Challenges.Free;
  Answers.Free;
end;

procedure TMainForm.WMUser(var msg: TMessage);
begin
  if not CanClose and (msg.WParam >= 0) and (msg.WParam < cNumHosts) then
    PingObjs[msg.WParam].AddPing(msg.LParam);
end;

function TPingTh.GetTimeout: Integer;
begin
  Result := PingSend.Timeout;
end;

procedure TPingTh.SetTimeout(value: Integer);
begin
  PingSend.Timeout := value;
end;

{ TPingObj }

procedure TPingObj.AddPing(const time: Integer);
var
  i: integer;
  Reset: Boolean;
begin
  if Time = -1 then
  begin
    Inc(PLCount);
    lblPing.Caption := '-';
    Inc(PLInRow);

    if CanReset and (PLInRow >= cNumTimeouts) then
    begin
      Reset := True;
      for i := 0 to cNumHosts-1 do
      begin
        if PingObjs[i].CanReset and (PingObjs[i].PLInRow < cNumTimeOuts) then
        begin
          Reset := False;
          break;
        end;
      end;

      if Reset then
      begin
        if WaitForSingleObject(hEvReset, 0) = WAIT_OBJECT_0 then
        begin
          if ThReset <> nil then ThReset.Free;
          ThReset := TThReset.Create;
        end;

        for i := 0 to cNumHosts-1 do
          PingObjs[i].PLInRow := 0;
      end;
    end;
  end
  else
  begin
    Inc(Count);
    Avg := (Avg * (Count - 1) + Time) / Count;
    lblAvg.Caption := IntToStr(Round(Avg)) + 'ms';
    lblPing.Caption := IntToStr(time) + 'ms';
    PLInRow := 0;
  end;
  lblPL.Caption := IntToStr(Round(PLCount / (PLCount + Count) * 100)) + '%';
end;

constructor TPingObj.Create(fPing, fAvg, fPL: TLabel; fID: integer);
begin
  lblPing := fPing;
  lblAvg := fAvg;
  lblPL := fPL;
  lblPing.Caption := '-';
  lblAvg.Caption := '-';
  lblPL.Caption := '-';
  Avg := 0;
  Count := 0;
  PLCount := 0;
  PLInRow := 0;
  ID := fID;
  CanReset := True;
end;

{ TThReset }

constructor TThReset.Create;
begin
  inherited Create(True);

  Telnet := TTelnetSend.Create;
  Telnet.TargetHost := ResetHost;
  Telnet.TargetPort := '23';
  Telnet.Timeout := 1000;
  Krok := 0;
  FreeOnTerminate := False;
  Resume;
end;

destructor TThReset.Destroy;
begin
  Telnet.Free;
end;

procedure TThReset.Execute;

  procedure Log(s: string);
  var
    LogFile: TFileStream;
    Mode: Cardinal;
  begin
    if FileExists(CurDir + cKomFileName) then
      Mode := fmOpenWrite
    else
      Mode := fmCreate;

    try
      LogFile := TFileStream.Create(CurDir + cKomFileName, Mode);
    except
      Exit;
    end;

    try
      s := s + #$0d#$0a;
      LogFile.Seek(0, soFromEnd);
      LogFile.Write(s[1], Length(s));
    finally
      LogFile.Free;
    end;
  end;

var
  str: string;
begin
  Start := GetTickCount;
  DeleteFile(CurDir + cKomFileName);
  Synchronize(MainForm.ResetStart);
  try
    try
      if not Telnet.Login then
      begin
        Synchronize(MainForm.ResetError);
        Exit;
      end;
    except
      begin
        Synchronize(MainForm.ResetError);
        Exit;
      end;
    end;

    try
      while not Terminated and (Krok < Challenges.Count) and
        (Start + cResetTimeout > GetTickCount) do
      begin
        try
          str := Telnet.RecvTerminated(Challenges[Krok]);
        except
          break;
        end;

        if str <> '' then
        begin
          Log('Step #' + IntToStr(Krok+1));
          Log(str + Challenges[Krok]);
          Log('-> ' + Answers[Krok]); 
          Telnet.Send(Answers[Krok] + #$0d#$0a);
          Inc(Krok);
        end;
      end;

      if Krok = Challenges.Count then
        Synchronize(MainForm.ResetOK)
      else
        Synchronize(MainForm.ResetProblem);
    finally
      Telnet.Logout;
    end;
  finally
    SetEvent(hEvReset);
  end;
end;

procedure TMainForm.ResetError;
var
  Text: string;
begin
  Text := 'Unable to reset, cannot connect to ' + ResetHost;
  StatusBar1.SimpleText := Text;
  Log(Text);
  CoolTrayIcon1.IconIndex := 2;
  Application.Icon.Assign(CoolTrayIcon1.Icon);
end;

procedure TMainForm.ResetOK;
var
  Text: string;
begin
  Text := 'Reset OK';
  StatusBar1.SimpleText := Text + ' - ' + DateTimeToStr(Now);
  Log(Text);
  CoolTrayIcon1.IconIndex := 0;
  Application.Icon.Assign(CoolTrayIcon1.Icon);
end;

procedure TMainForm.Log(str: string);
var
  LogFile: TFileStream;
  Mode: Cardinal;
begin
  if FileExists(CurDir + cLogFileName) then
    Mode := fmOpenWrite
  else
    Mode := fmCreate;

  try
    LogFile := TFileStream.Create(CurDir + cLogFileName, Mode);
  except
    Exit;
  end;

  try
    str := DateTimeToStr(Now) + ': ' + str + #$0d#$0a;
    LogFile.Seek(0, soFromEnd);
    LogFile.Write(str[1], Length(str));
  finally
    LogFile.Free;
  end;
end;

procedure TMainForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  if not CanClose then
  begin
    Action := caNone;
    CoolTrayIcon1.HideMainForm;
    Application.Icon.Assign(CoolTrayIcon1.Icon);
  end;
end;

procedure TMainForm.ResetStart;
begin
  StatusBar1.SimpleText := 'Trying to reset...';
  if CoolTrayIcon1.IconIndex = 0 then
  begin
    CoolTrayIcon1.IconIndex := 1;
    Application.Icon.Assign(CoolTrayIcon1.Icon);
  end;
end;

procedure TMainForm.ResetProblem;
var
  Text: string;
begin
  Text := 'Unable to reset, invalid response in step #' + IntToStr(ThReset.Krok+1);
  StatusBar1.SimpleText := Text;
  Log(Text);
  CoolTrayIcon1.IconIndex := 2;
  Application.Icon.Assign(CoolTrayIcon1.Icon);
end;

procedure TMainForm.Nastaven1Click(Sender: TObject);
begin
  Application.CreateForm(TSettingsForm, SettingsForm);
  SettingsForm.ShowModal;
  SettingsForm.Free;
end;

procedure TMainForm.Oprogramu1Click(Sender: TObject);
begin
  Application.CreateForm(TAboutForm, AboutForm);
  AboutForm.ShowModal;
  AboutForm.Free;
end;

end.
