#include <linux/init.h>
#include <linux/module.h>
#include <linux/netfilter.h>
#include <linux/netfilter_bridge.h>
#include <linux/netfilter_ipv4.h>
#include <linux/list.h>
#include <linux/if_ether.h>
#include <linux/etherdevice.h>
#include <net/netlink.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/smp.h>
#include <linux/spinlock.h>
#include <linux/kobject.h>
#include <linux/version.h>
#include <linux/ip.h>

#include "mtkmapfilter.h"
#include "load_balance.h"
#include "traffic_separation.h"
MODULE_LICENSE("Dual BSD/GPL");

/* drop default specific u32 ip addr */
#define SPECIFIC_IP_ADDR1 (htonl(0x08080808))			/* IP 8.8.8.8 */
#define SPECIFIC_IP_ADDR2 (htonl(0xD043DEDE))			/* IP 208.67.222.222 */

struct map_nt_entry {
	struct list_head list;
	unsigned char mac[6];
	unsigned char dev_addr[6];
	unsigned long updated;
	char intf[IFNAMSIZ + 1];
};

#define MAP_NETLINK 26
#define MAX_ENTRY_CNT 256
#define MAX_MSGSIZE 1024
#define NT_TIMEOUT (600 * HZ) /*10mins*/
#define TIMER_EXCUTE_PERIOD (5*HZ)
#define FDB_TIMEOUT (15*HZ)
#define DROP_TIMEOUT (180 * HZ)

#define CMDU_HLEN	8
#define TOPOLOGY_DISCOVERY	0x0000
#define VENDOR_SPECIFIC		0x0004

#define END_OF_TLV_TYPE 0

struct list_head nt[HASH_TABLE_SIZE];
spinlock_t nt_lock;
struct sock *nl_sk;
struct timer_list nt_timer;
struct timer_list dp_timer;
struct drop_specific_dest_ip_status drop_ip_status;
struct drop_specific_dest_ip_addr drop_ip_addr;
int nt_cnt;
unsigned char almac[ETH_ALEN];
unsigned char mtk_oui[3] = {0x00, 0x0C, 0xE7};

#define DIR_UNKNOWN		0x00
#define DIR_UPSTREAM 	0x01
#define DIR_DOWNSTREAM	0x02

struct hlist_head		dev_hash[HASH_TABLE_SIZE];
struct hlist_head		fdb_hash[HASH_TABLE_SIZE];

spinlock_t dev_lock;
spinlock_t fdb_lock;

unsigned char danamic_load_balance = 0;
unsigned char traffic_seperation = 0;

static inline struct map_net_device *get_map_net_dev_by_mac(unsigned char *mac);
char *dev_type2str(unsigned char type)
{
	switch(type) {
		case AP:
			return "AP";
		case APCLI:
			return "APCLI";
		case ETH:
			return "ETH";
		default:
			return "UNKNOWN";
	}
}

char *primary_type2str(unsigned char primary)
{
	switch(primary) {
		case INF_UNKNOWN:
			return "INF_NORMAL";
		case INF_PRIMARY:
			return "INF_PRIMARY";
		case INF_NONPRIMARY:
			return "INF_NONPRIMARY";
		default:
			return "UNKNOWN";
	}
}

bool eth_addr_equal(unsigned char *addr1, unsigned char *addr2)
{
	unsigned char ret = 0;
	ret =	((addr1[0] ^ addr2[0]) | (addr1[1] ^ addr2[1]) | (addr1[2] ^ addr2[2]) |
			(addr1[3] ^ addr2[3]) | (addr1[4] ^ addr2[4]) | (addr1[5] ^ addr2[5]));
	if (ret == 0) {
		return 1;
	}
	return 0;
}


bool is_zero_mac(unsigned char *mac)
{
	unsigned char zero_mac[6] = {0x00};

	return eth_addr_equal(mac, zero_mac);
}


void hex_dump_all(char *str, unsigned char *pSrcBufVA, unsigned int SrcBufLen)
{
	unsigned char *pt;
	int x;

	pt = pSrcBufVA;
	pr_info("%s: %p, len = %d\n",str,  pSrcBufVA, SrcBufLen);
	for (x=0; x<SrcBufLen; x++)
	{
		if (x % 16 == 0)
			pr_info("0x%04x : ", x);
		pr_info("%02x ", ((unsigned char)pt[x]));
		if (x%16 == 15) pr_info("\n");
	}
	pr_info("\n");
}


unsigned short get_1905_message_type(unsigned char *msg)
{
	unsigned char *pos = NULL;
	unsigned short mtype = 0;

	pos = msg;
	/*jump to message_type*/
	pos += 2;
	memcpy(&mtype, pos, 2);
	mtype = htons(mtype);

	return mtype;
}

int get_cmdu_tlv_length(unsigned char *buf)
{
    unsigned char *temp_buf = buf;
    unsigned short length = 0;

    temp_buf += 1;

    length = (*temp_buf);
    length = (length << 8) & 0xFF00;
    length = length |(*(temp_buf+1));

    return (length+3);
}

unsigned char is_vendor_specific_topology_discovery(struct sk_buff *skb)
{
	/*1905 internal using vendor specific tlv format
	   type			1 byte			0x11
	   length			2 bytes			0x, 0x
	   oui			3 bytes			0x00, 0x0C, 0xE7
	   function type	1 byte			0xff
	   suboui			3 bytes			0x00, 0x0C, 0xE7
	   sub func type	1 byte			0x				
	*/
	unsigned char *temp_buf = skb->data;
	unsigned char charactor_value[5] = {0xFF, 0x00, 0x0C, 0xE7, 0x00};
	int left_len = 0;
	
	left_len = (int)skb->len;
	left_len -= CMDU_HLEN;
	
	if (left_len < 11)
		return 0;
	
	temp_buf += CMDU_HLEN + 6;
	
	if ( !memcmp(temp_buf, charactor_value, sizeof(charactor_value)))
		return 1;
	else
		return 0;
}

int add_discovery_inf_vstlv(struct sk_buff *skb, unsigned char *rmac)
{
	unsigned char *temp_buf = skb->data;
	int length =0, left_len = 0;
	int offset = 0;
	
	/*vslen = tlvType(1 octet) + tlvLength(2 octets) + OUI(3 octets) +
	* subType(1 octet) + subLength(2 octets) + mac(6 octets)
	*/
	unsigned char vsinfo[15] = {0};

	/*build inf vs tlv*/
	vsinfo[0] = 11; /*Vendor-specific TLV value*/
	vsinfo[1] = 0x00;
	vsinfo[2] = 0x0c;
	memcpy(&vsinfo[3], mtk_oui, 3);
	vsinfo[6] = 0x00; /*subTyle inf mac*/
	vsinfo[7] = 0x00;
	vsinfo[8] = 0x06;
	memcpy(&vsinfo[9], rmac, 6);

	/*1905 cmdu header*/
	left_len = (int)skb->len;
	left_len -= CMDU_HLEN;
	temp_buf += CMDU_HLEN;
	offset += CMDU_HLEN;
	while (left_len > 0) {
        if (*temp_buf == END_OF_TLV_TYPE) {
            break;
        } else {
            /*ignore extra tlv*/
            length = get_cmdu_tlv_length(temp_buf);
            temp_buf += length;
			left_len -= length;
			offset += length;
        }
    }

	if (left_len < 0) {
		pr_info("%s error discovery msg\n", __func__);
		return -1;
	}


	/*vsinfo(15 octets) + End of message TLV(3 octets)*/
	if ((left_len + skb_tailroom(skb)) < 18) {
		/*do skb expand*/
		pr_info("%s not enough room for vstlv & end tlv\n", __func__);
		if (0 == pskb_expand_head(skb, 0, 15, GFP_ATOMIC)) {
			pr_info("%s expand room success\n", __func__);
			temp_buf = skb->data + offset;
			memcpy(temp_buf, vsinfo, 15);
			temp_buf += 15;
			memset(temp_buf, 0, 3);
			temp_buf += 3;
			skb_put(skb, 15);
		} else {
			pr_info("%s expand room fail drop!\n", __func__);
			return -1;
		}
	} else {
		if (left_len >= 18) {
			memcpy(temp_buf, vsinfo, 15);
			temp_buf += 15;
			memset(temp_buf, 0, 3);
		} else {
			memcpy(temp_buf, vsinfo, 15);
			temp_buf += 15;
			memset(temp_buf, 0, 3);
			temp_buf += 3;
			skb_put(skb, 15);
		}
	}

	return 0;
}

#if (LINUX_VERSION_CODE > KERNEL_VERSION(4,14,0))
void nt_timeout_func(struct timer_list *timer)
#else
void nt_timeout_func(unsigned long arg)
#endif
{
	int i;
	struct map_nt_entry *pos, *n;

	spin_lock_bh(&nt_lock);
	for (i = 0; i < HASH_TABLE_SIZE; i++) {
		list_for_each_entry_safe(pos, n, &nt[i], list) {
			if (time_after(jiffies, pos->updated+NT_TIMEOUT)) {
				pr_info("timeout! free %02x:%02x:%02x:%02x:%02x:%02x\n",
					pos->mac[0], pos->mac[1], pos->mac[2],
					pos->mac[3], pos->mac[4], pos->mac[5]);
				list_del(&pos->list);
				nt_cnt--;
				kfree(pos);
			}
		}
	}
	spin_unlock_bh(&nt_lock);
	mod_timer(&nt_timer, jiffies + TIMER_EXCUTE_PERIOD);

	return;
}

#if (LINUX_VERSION_CODE > KERNEL_VERSION(4,14,0))
void dp_timeout_func(struct timer_list *timer)
#else
void dp_timeout_func(unsigned long arg)
#endif
{
	/* stop drop the specific ip addr*/
	drop_ip_status.drop_flag = 0;
	pr_info("mapfilter:drop IP addr timeout! stop drop IP addr.");
	
	return;
}

int nt_get_intf(char *addr)
{
	int hash_idx = MAC_ADDR_HASH_INDEX(addr);
	struct map_nt_entry *pos;
	spin_lock_bh(&nt_lock);
	list_for_each_entry(pos, &nt[hash_idx], list) {
		if (eth_addr_equal(pos->mac, addr)) {
			memcpy(addr, pos->dev_addr, ETH_ALEN);
			spin_unlock_bh(&nt_lock);
			return 0;
		}
	}
	spin_unlock_bh(&nt_lock);
	memset(addr, 0 , ETH_ALEN);
	return -1;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
unsigned int _1905_hook_fn_pre_rout(void *priv,
			       struct sk_buff *skb,
			       const struct nf_hook_state *state)
#else
unsigned int _1905_hook_fn_pre_rout(unsigned int hooknum,
	struct sk_buff *skb,
	const struct net_device *in,
	const struct net_device *out,
	int (*okfn)(struct sk_buff *))
#endif
{
	const struct net_device *indev = NULL;
	struct ethhdr *hdr = eth_hdr(skb);
	int hash_idx = MAC_ADDR_HASH_INDEX(hdr->h_source);
	struct map_nt_entry *pos;
	unsigned short mtype = 0;
	unsigned char _1905_topo_discovery = 0;
	int ret = 0;
	
	/*step 1: if it's not a 1905 packet, ignore it.*/
	if (likely(skb->protocol != htons(0x893A)))
		return NF_ACCEPT;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
	indev = state->in;
#else
	indev = in;
#endif

	/*drop 1905 packets*/
	if (!memcmp(hdr->h_source, almac, ETH_ALEN))
		return NF_DROP;

	/*only handle skb without nonlinear memory*/
	if (likely(skb->data_len == 0)) {
		/*step 1.1: check if the pkts is a discovery message
		* need add vendor specific tlv specify receiving net device mac
		*/
		mtype = get_1905_message_type(skb->data);
		if (mtype == TOPOLOGY_DISCOVERY) {
			ret = add_discovery_inf_vstlv(skb, indev->dev_addr);
			if (ret < 0) {
				pr_info("%s drop error msg\n", __func__);
				return NF_DROP;
			}
			_1905_topo_discovery = 1;
		}
	}

	spin_lock_bh(&nt_lock);
	/*step 2: if the source address was add to the table, update it.*/
	list_for_each_entry(pos, &nt[hash_idx], list) {
		if (eth_addr_equal(pos->mac, hdr->h_source)) {
			strncpy(pos->intf, indev->name, IFNAMSIZ);
			memcpy(pos->dev_addr, indev->dev_addr, 6);
			pos->updated = jiffies;
			break;
		}
	}

	/*step 3: if the source address has not been added to the table.*/
	if (&pos->list == &nt[hash_idx] && nt_cnt <= MAX_ENTRY_CNT) {
		/*add a new entry to table.*/
		pos = kmalloc(sizeof(struct map_nt_entry), GFP_ATOMIC);
		if (pos == NULL)
			goto out;

		memset(pos, 0, sizeof(struct map_nt_entry));
		memcpy(pos->mac, hdr->h_source, 6);
		memcpy(pos->dev_addr, indev->dev_addr, 6);
		strncpy(pos->intf, indev->name, IFNAMSIZ);
		pos->updated = jiffies;

		pr_info("alloc new entry for %02x:%02x:%02x:%02x:%02x:%02x, interface:%s\n",
			hdr->h_source[0], hdr->h_source[1], hdr->h_source[2],
			hdr->h_source[3], hdr->h_source[4], hdr->h_source[5],
			indev->name);
		pr_info("recv intf mac %02x:%02x:%02x:%02x:%02x:%02x\n",
			indev->dev_addr[0], indev->dev_addr[1], indev->dev_addr[2],
			indev->dev_addr[3], indev->dev_addr[4], indev->dev_addr[5]);
		list_add_tail(&pos->list, &nt[hash_idx]);
		nt_cnt++;

	}
out:
	spin_unlock_bh(&nt_lock);
	if (!_1905_topo_discovery)
		return NF_REPEAT;
	
	return NF_ACCEPT;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
unsigned int map_hook_fn_pre_rout(void *priv,
				   struct sk_buff *skb,
				   const struct nf_hook_state *state)
#else
unsigned int map_hook_fn_pre_rout(unsigned int hooknum,
	struct sk_buff *skb,
	const struct net_device *in,
	const struct net_device *out,
	int (*okfn)(struct sk_buff *))
#endif
{
	struct ethhdr *hdr = eth_hdr(skb);
	const struct net_device *indev = NULL;
	struct map_net_device *in_map_dev = NULL;
	unsigned char device_type, primary_interface;
	int ret = NF_ACCEPT;
	unsigned char data_dir = DIR_UNKNOWN;

	unsigned int dest_ip;
	const struct iphdr *iph;
	
	/*
	a. process 1905 frame
	add. drop the specific IP packets to avoid two controller in auto-role
	b. process multi backhaul loop 
	c. process dynamic loading balance
	d. process traffic separation
	*/

	/*step a*/
	if (unlikely(skb->protocol == htons(0x893A))) {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
		ret = _1905_hook_fn_pre_rout(priv, skb, state);
#else
		ret = _1905_hook_fn_pre_rout(hooknum, skb, in, out, okfn);
#endif
		/*for 1905 topology discovery or any 1905 frame should be dropped*/
		if (ret == NF_ACCEPT || ret == NF_DROP)
			goto end;
		else if (ret == NF_REPEAT)
			ret = NF_ACCEPT;
	}
	
	/* add: drop the packets of specific IP addr to avoid two controller in auto-role */
	if (unlikely(drop_ip_status.drop_flag == 1)) {
		if (skb->protocol == htons(ETH_P_IP)) {
			iph = ip_hdr(skb);
			dest_ip = iph->daddr;
			if (drop_ip_addr.addr1 == dest_ip || drop_ip_addr.addr2 == dest_ip) {
				pr_info("mapfilter: drop the packet of specific IP addr: %08x \n", dest_ip);
				ret = NF_DROP;
				goto end;	
			}
		}
	}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
	indev = state->in;
#else
	indev = in;
#endif
	rcu_read_lock();
	in_map_dev = get_map_net_dev_by_mac(indev->dev_addr);
	if (!in_map_dev) {		
//		pr_info("%s-should not happen(in_map_dev %02x:%02x:%02x:%02x:%02x:%02x not found)\n", __func__, PRINT_MAC(indev->dev_addr));
		rcu_read_unlock();
		ret = NF_ACCEPT;
		goto end;
	}
	device_type = in_map_dev->inf.dev_type;
	primary_interface = in_map_dev->primary_interface;
	rcu_read_unlock();
	
	/*step b: drop multicast/broadcast packet which is not received from primary interface*/
	if ((is_multicast_ether_addr(hdr->h_dest) || is_broadcast_ether_addr(hdr->h_dest))) {
		if (device_type == APCLI && primary_interface == INF_NONPRIMARY) {
			ret = NF_DROP;
			goto end;
		}
	/*step c: learning for dynamic loading balance*/
	}
	if (danamic_load_balance) {
		if (device_type == AP)
			data_dir = DIR_UPSTREAM;
		else if (device_type == APCLI)
			data_dir = DIR_DOWNSTREAM;
		
		if (data_dir == DIR_UPSTREAM)
			ap_load_balance_learning(skb, (struct net_device*)indev);
		else if (data_dir == DIR_DOWNSTREAM)
			apcli_load_balance_learning(skb);
	}

	/*step d, traffic separation for ETH Mixed interface*/
	if (traffic_seperation) {
		 if (device_type == ETH)
			ret = ts_rx_process(skb);
	}

end:
	return ret;
}


#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
unsigned int map_hook_fn_forward(void *priv,
			       struct sk_buff *skb,
			       const struct nf_hook_state *state)
#else
unsigned int map_hook_fn_forward(unsigned int hooknum,
		struct sk_buff *skb,
		const struct net_device *in,
		const struct net_device *out,
		int (*okfn)(struct sk_buff *))
#endif
{
	struct ethhdr *hdr;	
	const struct net_device *indev = NULL, *outdev = NULL;
	struct map_net_device *in_map_dev = NULL, *out_map_dev = NULL;
	unsigned char data_dir;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
	indev = state->in;
	outdev = state->out;
#else
	indev = in;
	outdev = out;
#endif
	hdr = eth_hdr(skb);

	/*step 1: drop 1905 neighbor multicast packet*/
	if (unlikely(skb->protocol == htons(0x893A))) {
		if ((hdr->h_dest[0]&1) || !memcmp(hdr->h_dest, almac, ETH_ALEN))
			return NF_DROP;
	/*step 2:modify the data packet destination device*/
	}
	
	if (indev && indev->dev_addr && outdev && outdev->dev_addr) {	
		rcu_read_lock();
		out_map_dev = get_map_net_dev_by_mac(outdev->dev_addr);
		if (!out_map_dev) {
//			pr_info("%s-should not happen(out_map_dev %02x:%02x:%02x:%02x:%02x:%02x not found)\n", __func__, PRINT_MAC(outdev->dev_addr));
			goto accept;
		}

		data_dir = out_map_dev->inf.dev_type == APCLI ? DIR_UPSTREAM : DIR_UNKNOWN;

		in_map_dev = get_map_net_dev_by_mac(indev->dev_addr);

		if (!in_map_dev) {
//			pr_info("%s-should not happen(in_map_dev %02x:%02x:%02x:%02x:%02x:%02x not found)\n", __func__, PRINT_MAC(indev->dev_addr));
			goto accept;
		}

		if (in_map_dev->inf.dev_type != AP && out_map_dev->inf.dev_type == AP)
			data_dir = DIR_DOWNSTREAM;
		
		/*2.1 check the upstream data fisrt*/
		if (data_dir == DIR_UPSTREAM) {
		/*2.1.1 for multicast/broadcast packets, drop it if is not forwarded to the primary interface*/
			if ((is_multicast_ether_addr(hdr->h_dest) || is_broadcast_ether_addr(hdr->h_dest))) {
				if (out_map_dev->primary_interface == INF_NONPRIMARY)
					goto drop;
			/*2.1.2 for unicast packets, check if it is obey the forwarding rule*/
			} else {
				if (!in_map_dev->dest_dev)
					goto accept;
			
				/*if ethernt backhaul link exists, drop it, let it be forwarded by switch*/
				if (in_map_dev->inf.dev_type == ETH && in_map_dev->dest_dev == in_map_dev->dev) {
					goto drop;
				}

				/*if the packet received from apcli and send to apcli, drop it*/
				/*because we assuem the muilti Wifi backaul connect to same upstream device*/
				if (in_map_dev->inf.dev_type == APCLI) {
					goto drop;
				}
				
				skb->dev = in_map_dev->dest_dev;
				
				rcu_read_unlock();
				if (danamic_load_balance)
					apcli_load_balance_process(skb);
			}
		/*2.2 check the non-upstream data*/
		} else if (data_dir == DIR_DOWNSTREAM) {
			if (!is_multicast_ether_addr(hdr->h_dest) && !is_broadcast_ether_addr(hdr->h_dest)) {
				rcu_read_unlock();
				if (danamic_load_balance)
					ap_load_balance_process(skb);

			}
		}
	}

accept:
	rcu_read_unlock();
	return NF_ACCEPT;
drop:
	rcu_read_unlock();
	return NF_DROP;

}


#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
unsigned int map_hook_fn_local_out(void *priv,
			       struct sk_buff *skb,
			       const struct nf_hook_state *state)
#else
unsigned int map_hook_fn_local_out(unsigned int hooknum,
		struct sk_buff *skb,
		const struct net_device *in,
		const struct net_device *out,
		int (*okfn)(struct sk_buff *))
#endif
{
	struct ethhdr *hdr;
	const struct net_device *outdev = NULL;
	struct map_net_device *out_map_dev = NULL;
	unsigned char data_dir;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
	outdev = state->out;
#else
	outdev = out;
#endif

	hdr = eth_hdr(skb);

	if (outdev && outdev->dev_addr) {
		rcu_read_lock();
		out_map_dev = get_map_net_dev_by_mac(outdev->dev_addr);
		if (!out_map_dev) {
                  	rcu_read_unlock();
			goto accept;
		}

		data_dir = out_map_dev->inf.dev_type == AP ? DIR_DOWNSTREAM: DIR_UNKNOWN;
		rcu_read_unlock();

		if (data_dir == DIR_DOWNSTREAM) {
                  	/* only ucast pkt need load balance */
			if (!(hdr->h_dest[0] & 0x01)) {
				if (danamic_load_balance)
					ap_load_balance_process(skb);
			}
		}
	}

accept:
	return NF_ACCEPT;
}




#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
unsigned int map_hook_fn_local_out_post_routing(void *priv,
			       struct sk_buff *skb,
			       const struct nf_hook_state *state)
#else
unsigned int map_hook_fn_local_out_post_routing(unsigned int hooknum,
		struct sk_buff *skb,
		const struct net_device *in,
		const struct net_device *out,
		int (*okfn)(struct sk_buff *))
#endif
{
	struct ethhdr *hdr;	
	const struct net_device *outdev = NULL;
	struct map_net_device *out_map_dev = NULL;
	unsigned int ret = NF_ACCEPT;
	unsigned char device_type, primary_interface;
	
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
	outdev = state->out;
#else
	outdev = out;
#endif

	hdr = eth_hdr(skb);
	/*
	a. process the multi backhaul to avoid loop
	b. process traffic separation for Mixed Ethernet interface
	*/

	rcu_read_lock();
	
	out_map_dev = get_map_net_dev_by_mac(outdev->dev_addr);
	if (!out_map_dev) {
//		pr_info("%s-should not happen(out_map_dev %02x:%02x:%02x:%02x:%02x:%02x not found)\n", __func__, PRINT_MAC(outdev->dev_addr));
		rcu_read_unlock();
		goto end;
	}
	
	device_type = out_map_dev->inf.dev_type;
	primary_interface = out_map_dev->primary_interface;
	rcu_read_unlock();
	

	if ((is_multicast_ether_addr(hdr->h_dest) || is_broadcast_ether_addr(hdr->h_dest))) {
		if (device_type == APCLI && primary_interface == INF_NONPRIMARY) {
			ret = NF_DROP;
			goto end;	
		}
	}

	if (traffic_seperation) {
		if (device_type == ETH)
			ret = ts_tx_process(skb);
	}

end:
	return ret;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
unsigned int map_hook_fn_local_in(void *priv,
			       struct sk_buff *skb,
			       const struct nf_hook_state *state)
#else
unsigned int map_hook_fn_local_in(unsigned int hooknum,
	struct sk_buff *skb,
	const struct net_device *in,
	const struct net_device *out,
	int (*okfn)(struct sk_buff *))
#endif
{
	const struct net_device *indev = NULL;
	unsigned int ret = NF_ACCEPT;

	if (traffic_seperation) {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
	indev = state->in;
#else
	indev = in;
#endif
		ret = ts_local_in_process(skb);
	}

	return ret;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
unsigned int map_ts_hook_fn_local_out(void *priv,
			       struct sk_buff *skb,
			       const struct nf_hook_state *state)
#else
unsigned int map_ts_hook_fn_local_out(unsigned int hooknum,
	struct sk_buff *skb,
	const struct net_device *in,
	const struct net_device *out,
	int (*okfn)(struct sk_buff *))
#endif
{
	const struct net_device *outdev = NULL;
	struct map_net_device *out_map_dev = NULL;
	unsigned int ret = NF_ACCEPT;
	unsigned char device_type;

	if (traffic_seperation) {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
	outdev = state->out;
#else
	outdev = out;
#endif

		rcu_read_lock();
		out_map_dev = get_map_net_dev_by_mac(outdev->dev_addr);
		if (!out_map_dev) {
	//		pr_info("%s-should not happen(out_map_dev %02x:%02x:%02x:%02x:%02x:%02x not found)\n", __func__, PRINT_MAC(outdev->dev_addr));
			rcu_read_unlock();
			return ret;
		}

		device_type = out_map_dev->inf.dev_type;
		rcu_read_unlock();

		ret = ts_local_out_process(skb, device_type);
	}

	return ret;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,1,0))
unsigned int map_hook_fn_ip_pre_routing(void *priv,
			       struct sk_buff *skb,
			       const struct nf_hook_state *state)
#else
unsigned int map_hook_fn_ip_pre_routing(unsigned int hooknum,
	struct sk_buff *skb,
	const struct net_device *in,
	const struct net_device *out,
	int (*okfn)(struct sk_buff *))
#endif
{
	unsigned int ret = NF_ACCEPT;

	if (traffic_seperation) {
		ret = ts_ip_pre_routing_process(skb);
	}

	return ret;
}

void send_msg(char *msg, int pid)
{
	struct sk_buff *skb;
	struct nlmsghdr *nlh;
	int len = NLMSG_SPACE(MAX_MSGSIZE);

	if (!msg || !nl_sk)
		return;

	skb = alloc_skb(len, GFP_KERNEL);
	if (!skb) {
		pr_info("send_msg:alloc_skb error\n");
		return;
	}
	nlh = nlmsg_put(skb, 0, 0, 0, MAX_MSGSIZE, 0);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(3, 7, 0))
	NETLINK_CB(skb).pid = 0;
#else
	NETLINK_CB(skb).portid = 0;
#endif

	NETLINK_CB(skb).dst_group = 0;
	memcpy(NLMSG_DATA(nlh), msg, 6);
	netlink_unicast(nl_sk, skb, pid, MSG_DONTWAIT);
}

struct net_device *mt_dev_get_by_name(const char *name)
{
	struct net_device *dev;

	dev = dev_get_by_name(&init_net, name);

	if (dev)
		dev_put(dev);

	return dev;
}

static inline struct map_net_device *get_map_net_dev_by_mac(unsigned char *mac)
{
	struct map_net_device *dev = NULL;
	struct hlist_head *head = &dev_hash[MAC_ADDR_HASH_INDEX(mac)];
	
	hlist_for_each_entry_rcu(dev, head, hlist) {
		if (ether_addr_equal(dev->inf.mac, mac))
			break;
	}

	return dev;
}

static void free_map_net_device(struct rcu_head *head)
{
	struct map_net_device *dev = container_of(head, struct map_net_device, rcu);
	printk("----->free map device rcu %p\n", head);
	kfree(dev);
}

struct map_net_device *create_map_net_device(struct local_interface *inf, 
	struct net_device *dev, struct net_device *dest_dev, unsigned char primary)
{
	struct map_net_device *new_dev = NULL;

	new_dev = kmalloc(sizeof(struct map_net_device), GFP_ATOMIC);
	if (!new_dev) {
		pr_info("alloc new map_net_device fail\n");
		return NULL;
	}
	memset(new_dev, 0, sizeof(struct map_net_device));
	new_dev->dev = dev;
	new_dev->dest_dev = dest_dev;
	new_dev->primary_interface = primary;
	
	INIT_HLIST_NODE(&new_dev->hlist);
	
	memcpy(&new_dev->inf, inf, sizeof(struct local_interface));
	if (dev && dev->dev_addr)
		memcpy(new_dev->inf.mac, dev->dev_addr, ETH_ALEN);
	
	return new_dev;
}

int clear_map_net_device(void)
{
	int i = 0;
	struct hlist_head *head = NULL;
	struct map_net_device *dev = NULL;
	struct hlist_node *n = NULL;
	struct hlist_head clear_list;

	INIT_HLIST_HEAD(&clear_list);

	spin_lock_bh(&dev_lock);
	for(i = 0; i < HASH_TABLE_SIZE; i++) {
		head = &dev_hash[i];
		hlist_for_each_entry_safe(dev, n, head, hlist) {
			hlist_del_rcu(&dev->hlist);
			hlist_add_head_rcu(&dev->hlist, &clear_list);
		}
	}
	spin_unlock_bh(&dev_lock);
	
	hlist_for_each_entry_safe(dev, n, &clear_list, hlist) {
		hlist_del_rcu(&dev->hlist);
		synchronize_rcu();
		free_map_net_device(&dev->rcu);
	}
	return 0;
}

int update_map_net_device(struct local_itfs *dev_cfg)
{
	int i = 0;
	struct map_net_device *dev = NULL, *new_dev = NULL;
	struct net_device *net_dev = NULL;
	struct hlist_head *head = NULL;
	
	spin_lock_bh(&dev_lock);
	for (i = 0; i < dev_cfg->num; i++) {
		net_dev = mt_dev_get_by_name(dev_cfg->inf[i].name);
		if (!net_dev || (net_dev && !net_dev->dev_addr))
			continue;

		dev = get_map_net_dev_by_mac(net_dev->dev_addr);
		
		new_dev = create_map_net_device(&dev_cfg->inf[i], net_dev, NULL, INF_UNKNOWN);
				
		if (dev) {
			hlist_replace_rcu(&dev->hlist, &new_dev->hlist);			
			call_rcu(&dev->rcu, free_map_net_device);
		} else {
			head = &dev_hash[MAC_ADDR_HASH_INDEX(new_dev->inf.mac)];
			hlist_add_head_rcu(&new_dev->hlist, head);
		}
	}
	spin_unlock_bh(&dev_lock);

	return 0;
}


int set_primary_map_net_device(struct local_interface *inf, unsigned char primary)
{
	struct hlist_head *head = NULL;
	struct map_net_device *dev = NULL, *dev_exist = NULL, *dev_new = NULL;
	int i = 0;

	
	spin_lock_bh(&dev_lock);
	
	dev = get_map_net_dev_by_mac(inf->mac);

	if (!dev) {
		pr_info("fails to set primary interface(%s)\n", inf->name);	
		goto fail;
	}

	if (dev->inf.dev_type == AP || dev->inf.dev_type == ETH) {
		pr_info("cannot assign AP/ETH as (non)primary interface(%s)\n", inf->name);
		goto fail;
	}

	dev_new = create_map_net_device(&dev->inf, dev->dev, dev->dest_dev, primary);
	if (!dev_new)
		goto fail;

	hlist_replace_rcu(&dev->hlist, &dev_new->hlist);
	call_rcu(&dev->rcu, free_map_net_device);

	if (primary == INF_PRIMARY) {
	/*set other APCLI interfaces to non-primary interface*/
		for (i = 0; i < HASH_TABLE_SIZE; i++) {
			head = &dev_hash[i];
			
			hlist_for_each_entry_rcu(dev_exist, head, hlist) {
				if (dev_exist->inf.dev_type == APCLI && 
					memcmp(dev_new->inf.mac, dev_exist->inf.mac, ETH_ALEN))
					dev_exist->primary_interface = INF_NONPRIMARY;
			}
		}
	}
	
	spin_unlock_bh(&dev_lock);
	return 0;
fail:
	spin_unlock_bh(&dev_lock);
	return -1;
}

int set_uplink_path(struct local_interface *in, struct local_interface *out)
{
	struct map_net_device *dev_in = NULL, *dev_out = NULL;

	rcu_read_lock();
	dev_in = get_map_net_dev_by_mac(in->mac);

	if (!dev_in || ((dev_in) && dev_in->inf.dev_type == APCLI)) {
		goto fail;
	}
	if (out) {
		dev_out = get_map_net_dev_by_mac(out->mac);

		if (!dev_out || ((dev_out) && dev_out->inf.dev_type == AP)) {
			goto fail;
		}
		
		if (dev_in == dev_out) {
			goto fail;
		}
		
		dev_in->dest_dev = dev_out->dev;
	} else
		dev_in->dest_dev = NULL;

	rcu_read_unlock();
	return 0;
fail:
	rcu_read_unlock();
	return -1;
}

void dump_map_device_info(void)
{
	struct map_net_device *dev = NULL;
	struct hlist_head *head = NULL;
	int i = 0, count = 0;

	printk("[dump map device info]\n");
	printk("NAME\tMAC\tTYPE\tBAND\n");
	for (i = 0; i < HASH_TABLE_SIZE; i++) {
		head = &dev_hash[i];
		hlist_for_each_entry_rcu(dev, head, hlist) {
			printk("\t[%d]\n\t\t%s\t%02x:%02x:%02x:%02x:%02x:%02x\t%s\t%02x", count, 
				dev->inf.name, PRINT_MAC(dev->inf.mac),
				dev_type2str(dev->inf.dev_type), dev->inf.band);
			if (dev->dev && dev->dest_dev)
				printk("\t[uplink path] %s--->%s", dev->dev->name, dev->dest_dev->name);

			printk("\t[multicast primary] %s\n", primary_type2str(dev->primary_interface));
			count++;
		}
	}
}

void dump_fdb_info(void)
{
	struct mapfilter_fdb *fdb = NULL;
	struct hlist_head *head = NULL;
	int i = 0, count = 0;
	printk("[dump mapfilter fdb info]\n");
	printk("MAC\tRECV_DEV\tUPDATED\n");
	for (i = 0; i < HASH_TABLE_SIZE; i++) {
		head = &fdb_hash[i];
		hlist_for_each_entry_rcu(fdb, head, hlist) {
			printk("\t[%d]\n%02x:%02x:%02x:%02x:%02x:%02x\t%s\t%lu\n", count, PRINT_MAC(fdb->mac),
				fdb->dest_dev ? fdb->dest_dev->name : "N/A", fdb->updated);
			count++;
		}
	}
}
void dump_debug_info(void)
{
	dump_map_device_info();
	dump_fdb_info();
}

void update_apcli_link_status(struct apcli_link_status *status)
{
	struct map_net_device *dev = NULL;
	unsigned char band, link_status;
	rcu_read_lock();
	dev = get_map_net_dev_by_mac(status->in.mac);
	if (!dev) {
		pr_info("%s error interface %02x:%02x:%02x:%02x:%02x:%02x not found\n",
			__func__, PRINT_MAC(status->in.mac));
		goto end;
	}
	if (dev->inf.dev_type != APCLI) {
		pr_info("%s error %s is not apcli interface\n", __func__, dev->inf.name);
		goto end;
	}
	band = dev->inf.band;
	link_status = status->link_status;
	rcu_read_unlock();
	
	pr_info("update_apcli_link_status %s link %s\n", dev->inf.name, link_status ? "up" : "down");
	radio_link_change_update_apcli_session(band, link_status);

	return;
end:
	rcu_read_unlock();
	return;
}

int handle_map_command(struct map_netlink_message *cmd)
{
	int ret = 0;
	
	switch(cmd->type) {
		case UPDATE_MAP_NET_DEVICE:
			ret = update_map_net_device((struct local_itfs*)(cmd->event));
			break;
		case SET_PRIMARY_INTERFACE:
			{
				struct primary_itf_setting *psetting = (struct primary_itf_setting*)(cmd->event);
				ret = set_primary_map_net_device(&psetting->inf, psetting->primary);
			}
			break;
		case SET_UPLINK_PATH_ENTRY:
			{
				struct local_interface *in = NULL, *out = NULL;
				struct up_link_path_setting *psetting = (struct up_link_path_setting*)(cmd->event);

				in = &psetting->in;
				if (!is_zero_mac(psetting->out.mac))
					out = &psetting->out;
				
				printk("SET_UPLINK_PATH_ENTRY, in(%02x:%02x:%02x:%02x:%02x:%02x)",
					PRINT_MAC(in->mac));
				if (out)
					printk("-->out(%02x:%02x:%02x:%02x:%02x:%02x)\n",
						PRINT_MAC(out->mac));

				ret = set_uplink_path(in, out);		
			}
			break;
		case DUMP_DEBUG_INFO:
			dump_debug_info();
			dump_load_balance_debug_info();
			dump_ts_info();
			break;
		case UPDATE_APCLI_LINK_STATUS:
			{
				struct apcli_link_status *pstatus = (struct apcli_link_status*)(cmd->event);
				
				update_apcli_link_status(pstatus);
			}
			break;
		case UPDATE_CHANNEL_UTILIZATION:
			{
				struct radio_utilization *util = (struct radio_utilization*)(cmd->event);
				struct map_net_device *dev = NULL;
				unsigned char found = 0;
				int i = 0;

				rcu_read_lock();
				for (i = 0; i < HASH_TABLE_SIZE; i++) {
					hlist_for_each_entry_rcu(dev, &dev_hash[i], hlist) {
						if (dev->inf.dev_type == APCLI && dev->inf.band == util->band) {
							found = 1;
							break;
						}
					}

					if (found)
						break;
				}
				if (!dev) {
					pr_info("%s error apcli interface not found for band(%d)\n",
						__func__, util->band);
					rcu_read_unlock();
					break;
				}
				set_radio_channel_utilization(util, dev->dev);
				rcu_read_unlock();
			}
			break;
		case DYNAMIC_LOAD_BALANCE:
			{
				danamic_load_balance = cmd->event[0];
				pr_info("%s Dynamic Load Balance(%d)\n",
						__func__, danamic_load_balance);
			}
			break;
		case SET_TRAFFIC_SEPARATION_DEFAULT_8021Q:
			{
				struct ts_default_8021q *default_8021q = (struct ts_default_8021q*)(cmd->event);

				handle_ts_default_8021q(default_8021q->primary_vid, default_8021q->default_pcp);
			}
			break;
		case SET_TRAFFIC_SEPARATION_POLICY:
			{
				struct ts_policy *policy = (struct ts_policy *)(cmd->event);

				handle_ts_policy(policy);
			}
			break;
		case SET_DROP_SPECIFIC_IP_PACKETS_STATUS:
			{
				struct drop_specific_dest_ip_status *set_drop_ip_status = (struct drop_specific_dest_ip_status *)(cmd->event);

				drop_ip_status.drop_flag = set_drop_ip_status->drop_flag;
				if (set_drop_ip_status->drop_flag == 1) {
					mod_timer(&dp_timer, jiffies + DROP_TIMEOUT);
					pr_info("mapfilter: Start drop the specific ip packets now !!\n ");
				} else {
					pr_info("mapfilter: Stop drop the specific ip packets now !!\n ");
				}
			}
			break;
		case SET_TRANSPARENT_VID:
			{
				struct transparent_vids *trans_vids = (struct transparent_vids*)(cmd->event);

				handle_transparent_vlan(trans_vids);
			}
			break;
		case UPDATE_CLIENT_VID:
			{
				struct client_vid *client = (struct client_vid*)(cmd->event);

				handle_client_vid(client);
			}
			break;
		case TRAFFIC_SEPARATION_ONOFF:
			{
				traffic_seperation = cmd->event[0];
				pr_info("%s Traffic Seperation(%d)\n",
					__func__, traffic_seperation);
			}
			break;
		default:
			pr_info("unknown map command type(%02x)\n", cmd->type);
			break;
	}

	return ret;
}

void recv_nlmsg(struct sk_buff *skb)
{
	struct nlmsghdr *nlh = nlmsg_hdr(skb);
	struct map_netlink_message *msg = NULL;

	if (nlh->nlmsg_len < NLMSG_HDRLEN || skb->len < nlh->nlmsg_len)
		return;

	msg = (struct map_netlink_message *)NLMSG_DATA(nlh);
	handle_map_command(msg);
}
#if (!(LINUX_VERSION_CODE < KERNEL_VERSION(3, 6, 0)))
struct netlink_kernel_cfg nl_kernel_cfg = {
	.groups = 0,
	.flags = 0,
	.input = recv_nlmsg,
	.cb_mutex = NULL,
	.bind = NULL,
};
#endif

static struct nf_hook_ops map_ops[] = {
	{
		/*pre-rourting hook for multi backhaul*/	
		.hook		= map_hook_fn_pre_rout,
		.pf		= NFPROTO_BRIDGE,
		.hooknum	= NF_BR_PRE_ROUTING,
		.priority	= NF_BR_PRI_BRNF,
		//.owner		= THIS_MODULE,
	},
	{
		/*fowarding hook for multi backhaul*/
		.hook		= map_hook_fn_forward,
		.pf		= NFPROTO_BRIDGE,
		.hooknum	= NF_BR_FORWARD,
		.priority	= NF_BR_PRI_BRNF,
		//.owner		= THIS_MODULE,
	},
	{
		/*hook for mapr2 to strip vlan tag for local-in packet.*/
		.hook		= map_hook_fn_local_in,
		.pf		= NFPROTO_BRIDGE,
		.hooknum	= NF_BR_LOCAL_IN,
		.priority	= NF_BR_PRI_BRNF,
		//.owner		= THIS_MODULE,
	},
	{
		/*fowarding hook for local out*/
		.hook		= map_hook_fn_local_out,
		.pf		= NFPROTO_BRIDGE,
		.hooknum	= NF_BR_LOCAL_OUT,
		.priority	= NF_BR_PRI_FILTER_OTHER,
	},
	{
		/*fowarding hook for local out*/
		.hook		= map_ts_hook_fn_local_out,
		.pf		= NFPROTO_BRIDGE,
		.hooknum	= NF_BR_LOCAL_OUT,
		.priority	= NF_BR_PRI_FILTER_OTHER,
	},
	{
		.hook 		= map_hook_fn_local_out_post_routing,
		.pf			= NFPROTO_BRIDGE,
		.hooknum	= NF_BR_POST_ROUTING,
		.priority	= NF_BR_PRI_FILTER_OTHER,
	},	
	{
		.hook 		= map_hook_fn_ip_pre_routing,
		.pf			= NFPROTO_IPV4,
		.hooknum 	= NF_INET_PRE_ROUTING,
		.priority 	= NF_IP_PRI_FIRST,
	},
};

#define MAP_SO_BASE 1905
#define MAP_GET_MAC_BY_SRC MAP_SO_BASE
#define MAP_SET_ALMAC MAP_SO_BASE
#define MAP_SO_MAX (MAP_SO_BASE + 1)

static int do_map_set_ctl(struct sock *sk, int cmd,
		void __user *user, unsigned int len)
{
	int ret = 0;

	pr_info("do_map_set_ctl==>cmd(%d)\n", cmd);
	switch (cmd) {
	case MAP_SET_ALMAC:
		if (copy_from_user(almac, user, ETH_ALEN) != 0) {
			ret = -EFAULT;
			pr_info("do_map_set_ctl==>copy_from_user fail\n");
			break;
		}
		pr_info("do_map_set_ctl==>almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			almac[0],almac[1],almac[2],almac[3],almac[4],almac[5]);
		break;
	default:
		ret = -EINVAL;
		break;
	}

	return ret;
}

static int do_map_get_ctl(struct sock *sk, int cmd, void __user *user, int *len)
{
	int ret = 0;

	switch (cmd) {
	case MAP_GET_MAC_BY_SRC:
	{
		char addr[ETH_ALEN] = {0};

		if (*len < ETH_ALEN) {
			ret = -EINVAL;
			break;
		}

		if (copy_from_user(addr, user, ETH_ALEN) != 0) {
			ret = EFAULT;
			break;
		}

		nt_get_intf(addr);
		if (copy_to_user(user, addr, ETH_ALEN) != 0)
			ret = -EFAULT;
		break;
	}
	default:
		ret = -EINVAL;
	}

	return ret;
}
static struct nf_sockopt_ops map_sockopts = {
	.pf		= PF_INET,
	.set_optmin	= MAP_SO_BASE,
	.set_optmax	= MAP_SO_MAX,
	.set		= do_map_set_ctl,
#ifdef CONFIG_COMPAT
	.compat_set	= NULL,
#endif
	.get_optmin	= MAP_SO_BASE,
	.get_optmax	= MAP_SO_MAX,
	.get		= do_map_get_ctl,
#ifdef CONFIG_COMPAT
	.compat_get	= NULL,
#endif
	.owner		= THIS_MODULE,
};


static ssize_t map_nt_show(struct kobject *kobj,
		struct kobj_attribute *attr, char *buf)
{
	int i, cnt = 0;
	struct map_nt_entry *pos;

	spin_lock_bh(&nt_lock);
	for (i = 0; i < HASH_TABLE_SIZE; i++) {
		if (list_empty(&nt[i]))
			continue;
		list_for_each_entry(pos, &nt[i], list) {
			cnt += sprintf(&buf[cnt],
				"idx: %d\t%02x:%02x:%02x:%02x:%02x:%02x is at %s\n",
				i, pos->mac[0], pos->mac[1], pos->mac[2],
				pos->mac[3], pos->mac[4], pos->mac[5],
				pos->intf);
		}
	}
	spin_unlock_bh(&nt_lock);

	return cnt;
}

static ssize_t map_nt_cnt_show(struct kobject *kobj,
		struct kobj_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", nt_cnt);
}

static struct kobj_attribute map_sysfs_nt_show =
		__ATTR(nt_show, S_IRUGO, map_nt_show, NULL);
static struct kobj_attribute map_sysfs_nt_cnt_show =
		__ATTR(nt_cnt_show, S_IRUGO, map_nt_cnt_show, NULL);

static struct attribute *map_sysfs[] = {
	&map_sysfs_nt_show.attr,
	&map_sysfs_nt_cnt_show.attr,
	NULL,
};
static struct attribute_group map_attr_group = {
	.attrs = map_sysfs,
};
struct kobject *map_kobj;

#if (LINUX_VERSION_CODE > KERNEL_VERSION(4,12,0))
int nf_register_hook(struct nf_hook_ops *reg)
{
	struct net *net, *last;
	int ret;

	rtnl_lock();
	for_each_net(net) {
		ret = nf_register_net_hook(net, reg);
		if (ret && ret != -ENOENT)
			goto rollback;
	}
	rtnl_unlock();

	return 0;
rollback:
	last = net;
	for_each_net(net) {
		if (net == last)
			break;
		nf_unregister_net_hook(net, reg);
	}
	rtnl_unlock();
	return ret;
}

void nf_unregister_hook(struct nf_hook_ops *reg)
{
	struct net *net;

	rtnl_lock();
	for_each_net(net)
		nf_unregister_net_hook(net, reg);
	rtnl_unlock();
}

int nf_register_hooks(struct nf_hook_ops *reg, unsigned int n)
{
	unsigned int i;
	int err = 0;

	for (i = 0; i < n; i++) {
		err = nf_register_hook(&reg[i]);
		if (err)
			goto err;
	}
	return err;

err:
	if (i > 0)
		nf_unregister_hooks(reg, i);
	return err;
}

void nf_unregister_hooks(struct nf_hook_ops *reg, unsigned int n)
{
	while (n-- > 0)
		nf_unregister_hook(&reg[n]);
}
#endif 

static int __init map_init(void)
{
	int ret, i;

	spin_lock_init(&nt_lock);	
	spin_lock_init(&dev_lock);
	spin_lock_init(&fdb_lock);
	
	for (i = 0; i < HASH_TABLE_SIZE; i++) {
		INIT_LIST_HEAD(&nt[i]);
		INIT_HLIST_HEAD(&dev_hash[i]);
		INIT_HLIST_HEAD(&fdb_hash[i]);
	}
	
	load_balance_init();
	
	ts_init();

	drop_ip_status.drop_flag = 1;
	drop_ip_addr.addr1 = SPECIFIC_IP_ADDR1;
	drop_ip_addr.addr2 = SPECIFIC_IP_ADDR2;
	
	ret = nf_register_hooks(&map_ops[0], ARRAY_SIZE(map_ops));
	if (ret < 0) {
		pr_info("register nf hook fail, ret = %d\n", ret);
		goto error1;
	}

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 22))
	nl_sk = netlink_kernel_create(MAP_NETLINK, 0, recv_nlmsg, THIS_MODULE);
#elif (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 24))
	nl_sk = netlink_kernel_create(MAP_NETLINK, 0, recv_nlmsg, NULL, THIS_MODULE);
#elif (LINUX_VERSION_CODE < KERNEL_VERSION(3, 6, 0))
	nl_sk = netlink_kernel_create(&init_net, MAP_NETLINK, 0, recv_nlmsg, NULL, THIS_MODULE);
#elif (LINUX_VERSION_CODE < KERNEL_VERSION(3, 7, 0))
	nl_sk = netlink_kernel_create(&init_net, MAP_NETLINK, THIS_MODULE, &nl_kernel_cfg);
#else
	nl_sk = netlink_kernel_create(&init_net, MAP_NETLINK, &nl_kernel_cfg);
#endif
	if (!nl_sk) {
		pr_info("create netlink socket error.\n");
		ret = -EFAULT;
		goto error2;
	}

	ret = nf_register_sockopt(&map_sockopts);
	if (ret < 0)
		goto error3;

	map_kobj = kobject_create_and_add("mapfilter", NULL);
	if (!map_kobj) {
		ret = -EFAULT;
		goto error4;
	}

	ret = sysfs_create_group(map_kobj, &map_attr_group);
	if (ret)
		goto error5;

#if (LINUX_VERSION_CODE > KERNEL_VERSION(4,14,0))
	timer_setup(&nt_timer, nt_timeout_func, 0);
	timer_setup(&dp_timer, dp_timeout_func, 0); 
#else
	init_timer(&nt_timer);
	nt_timer.function = nt_timeout_func;
	init_timer(&dp_timer);
	dp_timer.function = dp_timeout_func;
#endif
	nt_timer.expires = jiffies + TIMER_EXCUTE_PERIOD;
	add_timer(&nt_timer);
	dp_timer.expires = jiffies + DROP_TIMEOUT;
	add_timer(&dp_timer);

	return ret;
error5:
	kobject_put(map_kobj);
error4:
	nf_unregister_sockopt(&map_sockopts);
error3:
	sock_release(nl_sk->sk_socket);
error2:
	nf_unregister_hooks(&map_ops[0], ARRAY_SIZE(map_ops));
error1:
	load_balance_deinit();
	return ret;
}

static void __exit map_exit(void)
{
	int i;
	struct map_nt_entry *pos, *n;

	del_timer_sync(&nt_timer);
	del_timer_sync(&dp_timer);
  
	sysfs_remove_group(map_kobj, &map_attr_group);
	kobject_put(map_kobj);

	nf_unregister_sockopt(&map_sockopts);

	if (nl_sk != NULL)
		sock_release(nl_sk->sk_socket);

	nf_unregister_hooks(&map_ops[0], ARRAY_SIZE(map_ops));

	for (i = 0; i < HASH_TABLE_SIZE; i++) {
		list_for_each_entry_safe(pos, n, &nt[i], list) {
			pr_info("exit! free %02x:%02x:%02x:%02x:%02x:%02x\n",
				pos->mac[0], pos->mac[1], pos->mac[2],
				pos->mac[3], pos->mac[4], pos->mac[5]);
			list_del(&pos->list);
			nt_cnt--;
			kfree(pos);
		}
	}
	
	load_balance_deinit();

	ts_deinit();

	clear_map_net_device();

	printk("module exit\n");
	return;
}

module_init(map_init);
module_exit(map_exit);

