/*
 * cleopatre/application/p1905_managerd/src/cmdu_tlv_parse.c
 *
 * (C) Copyright 2013 MSsar Semiconductor, Inc.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <syslog.h>
#include <assert.h>
#include <linux/if_ether.h>
#include <asm/byteorder.h>
#include "cmdu_tlv_parse.h"
#include "byteorder.h"
#include "debug.h"
#include "p1905_utils.h"

unsigned char mtk_oui[3] = {0x00, 0x0C, 0xE7};

#define TOTAL_AL_MAC_ADDR_TLV_LENGTH 9
#define TOTAL_MAC_ADDR_TLV_LENGTH 9

extern WSC_ATTR_STATUS parse_wsc_msg(
    struct p1905_managerd_ctx *ctx, unsigned char *pkt, unsigned short length);
extern int wapp_set_info_by_msgtype(struct p1905_managerd_ctx *ctx, unsigned short msgtype, unsigned char *bssid,
	void *data, int datalen);


int get_cmdu_tlv_length(unsigned char *buf)
{
    unsigned char *temp_buf = buf;
    unsigned short length = 0;

    temp_buf +=1;//shift to length field

    length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

    return (length+3);
}

int parse_al_mac_addr_type_tlv(unsigned char *buf,unsigned char *al_mac)
{
    unsigned char *temp_buf;
	unsigned short len = 0;

    temp_buf = buf;

    if((*temp_buf) == AL_MAC_ADDR_TLV_TYPE)
        temp_buf++;
    else
        return -1;

	len = *(unsigned short *)temp_buf;
	len = be2cpu16(len);

    if (len == 0x6)
        temp_buf += 2;
    else
        return -1;

    memcpy(al_mac,temp_buf,ETH_ALEN);

    return TOTAL_AL_MAC_ADDR_TLV_LENGTH;
}


int parse_mac_addr_type_tlv(unsigned char *buf,unsigned char *mac)
{
    unsigned char *temp_buf;
	unsigned short len = 0;

    temp_buf = buf;

    if((*temp_buf) == MAC_ADDR_TLV_TYPE)
        temp_buf++;
    else
        return -1;

	len = *(unsigned short *)temp_buf;
	len = be2cpu16(len);

    if (len == 0x6)
        temp_buf += 2;
    else
        return -1;

    memcpy(mac,temp_buf,ETH_ALEN);

    return TOTAL_MAC_ADDR_TLV_LENGTH;
}

int parse_vs_tlv(unsigned char *buf, struct vs_value *vs_info)
{
    unsigned char *temp_buf = buf;
	unsigned short length = 0, sublen = 0;
	unsigned char func_type = 0, sub_type = 0;
	unsigned char suboui[3] = {0x00, 0x0C, 0xE7};

    if ((*temp_buf) == VENDOR_SPECIFIC_TLV_TYPE)
        temp_buf++;
    else
        return -1;

    length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);
	temp_buf += 2;
	if (memcmp(temp_buf, mtk_oui, 3)) {
        debug(DEBUG_ERROR, "OUI not mtk_oui; do not handle!\n");
		return (length + 3);
	}
	temp_buf += 3;
	func_type = *temp_buf++;

	switch(func_type) {
	case 0: /*mac sub type*/
		if (length != 12) {
			debug(DEBUG_ERROR, "error vs mac tlv(%d)!\n", length);
			return -1;
		}
		sublen = *(unsigned short *)temp_buf;
		sublen = be2cpu16(sublen);
		if (sublen != ETH_ALEN) {
			debug(DEBUG_ERROR, "error mac len!\n");
			return -1;
		}
		temp_buf += 2;
    	memcpy(vs_info->itf_mac, temp_buf, ETH_ALEN);
        debug(DEBUG_TRACE, "mac(%02x:%02x:%02x:%02x:%02x:%02x)!\n",
			PRINT_MAC(vs_info->itf_mac));
		break;
	case 0xff:
		if (os_memcmp(suboui, temp_buf, sizeof(suboui))) {
			debug(DEBUG_ERROR, "error vendor specific tlv charactor_value\n");
			return -1;
		}
		temp_buf += sizeof(suboui);
		sub_type = *temp_buf++;

		switch (sub_type) {
		case 0x00:
			if (length != 8) {
				debug(DEBUG_ERROR, "error vs discovery tlv(%d)!\n", length);
				return -1;
			}
			vs_info->discovery = 1;
			break;
		case 0x01:
			if (length != 16) {
				debug(DEBUG_ERROR, "error vs notification tlv(%d)!\n", length);
				return -1;
			}
			sublen = *(unsigned short *)temp_buf;
			sublen = be2cpu16(sublen);
			temp_buf += 2;
			if (sublen != ETH_ALEN) {
				debug(DEBUG_ERROR, "error almac len!\n");
				return -1;
			}
			os_memcpy(vs_info->neth_almac, temp_buf, ETH_ALEN);
		}
		break;
	default:
		break;
	}

	return (length + 3);
}


int parse_device_info_type_tlv(struct p1905_managerd_ctx *ctx, unsigned char *buf,
                               struct list_head_devinfo *devinfo_head)
{
    unsigned char *temp_buf;
    unsigned short length = 0;
    unsigned char itfs_num=0;
    unsigned char vs_info_len =0;
    int i =0;
    struct device_info_db *dev;
    unsigned char mac[ETH_ALEN];
    unsigned char almac[ETH_ALEN] = {0};
	unsigned char topo_bss_num = 0;

    temp_buf = buf;

    if((*temp_buf) == DEVICE_INFO_TLV_TYPE)
    {
        temp_buf++;
    }
    else
    {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

    //shift to tlv value field
    temp_buf+=2;

    //skip AL MAC address field
    memcpy(almac, temp_buf, ETH_ALEN);
    temp_buf+=ETH_ALEN;

    //get the amount of local interface
    itfs_num = *temp_buf;
    temp_buf +=1;

    for(i=0;i<itfs_num;i++)
    {
        memcpy(mac,temp_buf,ETH_ALEN);
        temp_buf += ETH_ALEN;

        dev = (struct device_info_db *)malloc(sizeof(struct device_info_db));
		if (dev == NULL)
			return -1;
        memcpy(dev->mac_addr,mac,ETH_ALEN);
        /*init the list head of p1905.1 neighbor device list*/
        SLIST_INIT(&(dev->p1905_nbrdb_head));
        /*init the list head of non-p1905.1 neighbor device list*/
        SLIST_INIT(&(dev->non_p1905_nbrdb_head));
        SLIST_INSERT_HEAD(devinfo_head, dev, devinfo_entry);

        //debug("device mac address = 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x\n",
        //       dev->mac_addr[0], dev->mac_addr[1], dev->mac_addr[2],
        //       dev->mac_addr[3],dev->mac_addr[4],dev->mac_addr[5]);

        dev->media_type = (*(unsigned short *)temp_buf);
        dev->media_type = cpu2be16(dev->media_type);
        temp_buf +=2;
        //debug("dev->media_type = %d",dev->media_type);

        vs_info_len = (*temp_buf);
        dev->vs_info_len = vs_info_len;
        temp_buf +=1;

        //debug("vs_info_len = %d\n",vs_info_len);
        if(vs_info_len > 0)
        {
           dev->vs_info = (unsigned char*)malloc(vs_info_len);
	    if(dev->vs_info ==  NULL)
	        return -1;
           memcpy(dev->vs_info, temp_buf, vs_info_len);
           temp_buf += vs_info_len;

		   if (vs_info_len == 10 && (*(dev->vs_info + 6) == 0x00))
		   		topo_bss_num++;
        }
    }

    return (length+3);
}

int parse_bridge_capability_type_tlv(unsigned char *buf,
                                struct list_head_brcap *brcap_head)
{
    unsigned char *temp_buf;
    unsigned short length = 0;
    unsigned char br_num=0;
    int i = 0;
    unsigned char br_itfs_num=0;
    struct device_bridge_capability_db *br;

    temp_buf = buf;

    if((*temp_buf) == BRIDGE_CAPABILITY_TLV_TYPE)
    {
	    temp_buf++;
    }
    else
    {
	    return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    //shift to tlv value field
    temp_buf+=2;

    //the field of total number of bridge tuple
    br_num = *temp_buf;
	temp_buf++;

    //debug("br_num = %d\n",br_num);
    //loop br_num times to store the bridge tuple information
    for(i=0;i<br_num;i++)
    {
        br_itfs_num = *temp_buf;
        temp_buf++;

       // debug("br_itfs_num = %d\n",br_itfs_num);

        br= (struct device_bridge_capability_db *)malloc(sizeof(struct device_bridge_capability_db));
	 if (br == NULL)
	     return -1;
        br->interface_amount = br_itfs_num;

        if(br_itfs_num > 0)
        {
            br->interface_mac_tuple = (unsigned char *)malloc(br_itfs_num*ETH_ALEN);
	     if (br->interface_mac_tuple == NULL)
	     {
	         free(br);
		  return -1;			
	     }
            memcpy(br->interface_mac_tuple,temp_buf,(br_itfs_num*ETH_ALEN));
	    }

        LIST_INSERT_HEAD(brcap_head, br, brcap_entry);

        temp_buf += br_itfs_num * ETH_ALEN;

        //debug("br->interface_amount = %d\n",br->interface_amount);
    }

    return (length+3);
}

int parse_p1905_neighbor_device_type_tlv(unsigned char *buf,
    struct list_head_devinfo *devinfo_head)
{
    unsigned char *temp_buf;
    unsigned short length = 0;
    struct p1905_neighbor_device_db *dev;
    struct device_info_db *dev_info;
    unsigned char local_mac[ETH_ALEN];
    unsigned char al_mac[ETH_ALEN];
    unsigned char exist = 0;
    unsigned char new_db = 1;
    int tuple_num = 0;
    int i = 0;

    temp_buf = buf;

    if((*temp_buf) == P1905_NEIGHBOR_DEV_TLV_TYPE)
    {
        temp_buf++;
    }
    else
    {
        return -1;
    }

    /*calculate tlv length*/
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    /*shift to tlv value field*/
    temp_buf+=2;

    memcpy(local_mac,temp_buf,ETH_ALEN);
    temp_buf+=ETH_ALEN;

    /* tlv value has one local mac address field(6 bytes) and several
     * (al mac address field(6 bytes) + 802.1 exist field(1 byte)),
     * so tlv (value length - 6 ) % 7 must be zero
     */
    if(((length - 6) % 7) == 0)
    {
        tuple_num = (length - 6) /7;
        //debug("tuple_num = %d\n",tuple_num);
    }
    else
    {
        //debug("ignore p1905.1 neighbor tlv\n");
        return (length+3);
    }

    SLIST_FOREACH(dev_info, devinfo_head, devinfo_entry)
    {
        if(!memcmp(local_mac, dev_info->mac_addr, ETH_ALEN))
        {
            exist = 1;
            break;
        }
    }

    if(exist)
    {
        for(i=0;i<tuple_num;i++)
        {
            memcpy(al_mac, temp_buf, ETH_ALEN);
            temp_buf += ETH_ALEN;

            new_db = 1;

            /* maybe we don't needt to do this check because the old topology
             * response db will be deleted when we get a new topology response
             * message.
             */
            SLIST_FOREACH(dev, &(dev_info->p1905_nbrdb_head), p1905_nbrdb_entry)
            {
                if(!memcmp(al_mac, dev->p1905_neighbor_al_mac, ETH_ALEN))
                {
                    new_db = 0;
                    break;
                }
            }

            if(new_db)
            {
                dev = (struct p1905_neighbor_device_db *)malloc(sizeof(struct p1905_neighbor_device_db));
		        if(dev == NULL)
		  	        return -1;
                memcpy(dev->p1905_neighbor_al_mac, al_mac, ETH_ALEN);
                SLIST_INSERT_HEAD(&(dev_info->p1905_nbrdb_head), dev, p1905_nbrdb_entry);
            }

            if(((*temp_buf) & 0x80) == 0x80)
                dev->ieee_802_1_bridge_exist = 1;
            else
                dev->ieee_802_1_bridge_exist = 0;

            temp_buf += 1;
        }
    }

    return (length+3);
}

int parse_non_p1905_neighbor_device_type_tlv(unsigned char *buf,
    struct list_head_devinfo *devinfo_head)
{
    unsigned char *temp_buf;
    unsigned short length = 0;
    int tuple_num = 0;
    unsigned char local_mac[ETH_ALEN];
    unsigned char neighbor_mac[ETH_ALEN];
    struct device_info_db *dev_info;
    struct non_p1905_neighbor_device_list_db *dev;

    unsigned char exist = 0;
    unsigned char new_db = 1;
    int i = 0;

    temp_buf = buf;

    if((*temp_buf) == NON_P1905_NEIGHBOR_DEV_TLV_TYPE)
        temp_buf++;
    else
    {
        return -1;
    }

    /*calculate tlv length*/
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    /*shift to tlv value field*/
    temp_buf+=2;

    /*get local interface field*/
    memcpy(local_mac,temp_buf,ETH_ALEN);
    temp_buf+=ETH_ALEN;

    /*calculate how many non-1905.1 device*/
    if(((length - 6) % 6) == 0)
    {
        tuple_num = (length - 6) /6;
        //debug("tuple_num = %d\n",tuple_num);
    }
    else
    {
        //debug("ignore non p1905.1 neighbor tlv\n");
        return (length+3);
    }

    SLIST_FOREACH(dev_info, devinfo_head, devinfo_entry)
    {
        if(!memcmp(local_mac,dev_info->mac_addr,ETH_ALEN))
        {
            exist = 1;
            break;
        }
    }

    if(exist)
    {
        for(i=0;i<tuple_num;i++)
        {
            memcpy(neighbor_mac, temp_buf, ETH_ALEN);
            temp_buf += ETH_ALEN;

            new_db = 1;
            /* maybe we don't need to do this check because the old topology
             * response db will be deleted when we get a new topology response
             * message.
             */
            SLIST_FOREACH(dev, &(dev_info->non_p1905_nbrdb_head), non_p1905_nbrdb_entry)
            {
                if(!memcmp(neighbor_mac, dev->non_p1905_device_interface_mac,ETH_ALEN))
                {
                    new_db = 0;
                    break;
                }
            }
            if(new_db)
            {
                dev = (struct non_p1905_neighbor_device_list_db *)\
                        malloc(sizeof(struct non_p1905_neighbor_device_list_db));
				if (dev == NULL)
					return -1;
                memcpy(dev->non_p1905_device_interface_mac, neighbor_mac, ETH_ALEN);
                SLIST_INSERT_HEAD(&(dev_info->non_p1905_nbrdb_head), dev, non_p1905_nbrdb_entry);
            }
        }
    }
    return (length+3);
}

int parse_vendor_specific_type_tlv(unsigned char *buf,struct p1905_vs_info *vs_info)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == VENDOR_SPECIFIC_TLV_TYPE)
        temp_buf++;
    else
    {
        return -1;
    }

    /*calculate tlv length*/
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    /*shift to tlv value field*/
	temp_buf+=2;

    return (length+3);
}

int parse_link_metric_query_type_tlv(unsigned char *buf,
                        unsigned char *target, unsigned char *type)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == LINK_METRICS_QUERY_TLV_TYPE)
        temp_buf++;
    else
    {
        return -1;
    }

    /*calculate tlv length*/
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    /*shift to tlv value field*/
    temp_buf+=2;

    /*set target mac = 0 if query all neighbors*/
    if(*temp_buf == QUERY_ALL_NEIGHBOR) {
        memset(target, 0, ETH_ALEN);
        temp_buf+=1;

        /*add for IOT issue with BRCM*/
        if(length == 8)
            temp_buf+=ETH_ALEN;

    } else if(*temp_buf == QUERY_SPECIFIC_NEIGHBOR) {
        temp_buf+=1;
        memcpy(target, temp_buf, ETH_ALEN);
        temp_buf+=ETH_ALEN;
    } else {
    	debug(DEBUG_ERROR, "reserve valus %d\n", *temp_buf);
		return -1;
    }

    if(*temp_buf == TX_METRICS_ONLY)
        *type = TX_METRICS_ONLY;
    else if(*temp_buf == RX_METRICS_ONLY)
        *type = RX_METRICS_ONLY;
    else if(*temp_buf == BOTH_TX_AND_RX_METRICS)
        *type = BOTH_TX_AND_RX_METRICS;

    return (length+3);
}

#ifdef SUPPORT_ALME
int parse_transmitter_link_metrics_type_tlv(unsigned char *buf,
            unsigned char *local_al_mac, ALME_GET_METRIC_RSP *metric_rsp)
{
    unsigned char *temp_buf;
    int length = 0;
    unsigned char response_al_mac[6];
    unsigned short link_tuple_num = 0;
    int i = 0;
    unsigned int value;


    temp_buf = buf;

    if((*temp_buf) == TRANSMITTER_LINK_METRIC_TYPE)
        temp_buf++;
    else
    {
        return -1;
    }

    /*calculate tlv length*/
    length = (*temp_buf);
    length = (length << 8) & 0xFF00;
    length = length |(*(temp_buf+1));
    /*shift to tlv value field*/
    temp_buf+=2;

    /*transmit device's al mac field*/
    memcpy(response_al_mac, temp_buf, 6);
    temp_buf += 6;

    /*shift to neighbor al mac field*/
    /*if the neighbor al mac is not local, return fail*/
    if(memcmp(local_al_mac, temp_buf, 6))
    {
        debug(DEBUG_ERROR, "got a wrong target tx link metrics tlv\n");
        return -1;
    }
    temp_buf += 6;

    if(((length - 12) % 29) != 0)
    {
        debug(DEBUG_ERROR, "got a wrong length tx link metrics tlv\n");
        return -1;
    }

    link_tuple_num = (length - 12) /29;

    if(link_tuple_num > MAX_1905_LINK_NUM)
    {
        debug(DEBUG_ERROR, "got a wrong link num tx link metrics tlv\n");
        return -1;
    }

    metric_rsp->descriptor_num = link_tuple_num;
    for(i=0;i<link_tuple_num;i++)
    {
        /*fill into neighbor al mac field first*/
        memcpy(metric_rsp->descriptorlist[i].neighbor_al_mac, response_al_mac, 6);

        /*shift to interface mac addr of neighbor field*/
        temp_buf += 6;
        /*fill into local interface field in alme*/
        memcpy(metric_rsp->descriptorlist[i].local_itf_mac, temp_buf, 6);
        temp_buf += 6;

        /*intf type*/
        value = *(temp_buf);
        value = ((value << 8) & 0x0000FF00)| (*(temp_buf + 1));
        metric_rsp->descriptorlist[i].link_metrics.itf_type = (unsigned short)value;
        temp_buf += 2;

        /*ieee802.1 bridge*/
        metric_rsp->descriptorlist[i].ieee802_1_bridge = *(temp_buf);
        temp_buf += 1;

        /*packet error*/
        value = *(temp_buf);
        value = ((value << 8) & 0x0000FF00)| (*(temp_buf + 1));
        value = ((value << 8) & 0x00FFFF00)| (*(temp_buf + 2));
        value = ((value << 8) & 0xFFFFFF00)| (*(temp_buf + 3));
        metric_rsp->descriptorlist[i].link_metrics.tx_packet_error = value;
        temp_buf += 4;

        /*transmitted packets*/
        value = *(temp_buf);
        value = ((value << 8) & 0x0000FF00)| (*(temp_buf + 1));
        value = ((value << 8) & 0x00FFFF00)| (*(temp_buf + 2));
        value = ((value << 8) & 0xFFFFFF00)| (*(temp_buf + 3));
        metric_rsp->descriptorlist[i].link_metrics.tx_total_packet = value;
        temp_buf += 4;

        /*mac throghput capacity*/
        value = *(temp_buf);
        value = ((value << 8) & 0x0000FF00)| (*(temp_buf + 1));
        metric_rsp->descriptorlist[i].link_metrics.max_throughput_capacity = (unsigned short)value;
        temp_buf += 2;

        /*link availability*/
        value = *(temp_buf);
        value = ((value << 8) & 0x0000FF00)| (*(temp_buf + 1));
        metric_rsp->descriptorlist[i].link_metrics.link_availability = (unsigned short)value;
        temp_buf += 2;

        /*phy rate*/
        value = *(temp_buf);
        value = ((value << 8) & 0x0000FF00)| (*(temp_buf + 1));
        metric_rsp->descriptorlist[i].link_metrics.phy_rate = (unsigned short)value;
        temp_buf += 2;
    }

    return (length+3);
}

int parse_receiver_link_metrics_type_tlv(unsigned char *buf,
                unsigned char *local_al_mac, ALME_GET_METRIC_RSP *metric_rsp)
{
    unsigned char *temp_buf;
    int length = 0;
    unsigned char response_al_mac[6];
    unsigned short link_tuple_num = 0;
    int i = 0, j = 0;
    unsigned int value;

    temp_buf = buf;

    if((*temp_buf) == RECEIVER_LINK_METRIC_TYPE)
        temp_buf++;
    else
    {
        return -1;
    }

    /*calculate tlv length*/
    length = (*temp_buf);
    length = (length << 8) & 0xFF00;
    length = length |(*(temp_buf+1));
    /*shift to tlv value field*/
    temp_buf+=2;

    /*transmit device's al mac field*/
    memcpy(response_al_mac, temp_buf, 6);
    temp_buf += 6;

    /*shift to neighbor al mac field*/
    /*if the neighbor al mac is not local, return fail*/
    if(memcmp(local_al_mac, temp_buf, 6))
    {
        debug(DEBUG_ERROR, "got a wrong target rx link metrics tlv\n");
        return -1;
    }
    temp_buf += 6;

    if(((length - 12) % 23) != 0)
    {
        debug(DEBUG_ERROR, "got a wrong length rx link metrics tlv\n");
        return -1;
    }

    link_tuple_num = (length - 12) /23;

    /*compare link tuple num with tx and max link number*/
    if((link_tuple_num > MAX_1905_LINK_NUM) || (link_tuple_num != metric_rsp->descriptor_num))
    {
        debug(DEBUG_ERROR, "got a wrong link num rx link metrics tlv %d\n",link_tuple_num);
        return -1;
    }

    for(i=0;i<link_tuple_num;i++)
    {
        /*shift to interface mac addr of neighbor field*/
        temp_buf += 6;
        /*get correct position in metric_rsp*/
        for(j=0;j<link_tuple_num;j++)
        {
            if(!memcmp(temp_buf, metric_rsp->descriptorlist[j].local_itf_mac, 6))
                break;
        }
        if(j == link_tuple_num)
        {
            debug(DEBUG_ERROR, "got un-matched neighbor interface in rx link metric tlv\n");
            return -1;
        }
        temp_buf += 6;
        /*ignore iftype field*/
        temp_buf += 2;
        /*packet error*/
        value = *(temp_buf);
        value = ((value << 8) & 0x0000FF00)| (*(temp_buf + 1));
        value = ((value << 8) & 0x00FFFF00)| (*(temp_buf + 2));
        value = ((value << 8) & 0xFFFFFF00)| (*(temp_buf + 3));
        metric_rsp->descriptorlist[j].link_metrics.rx_packet_error = value;
        temp_buf += 4;
        /*total packet*/
        value = *(temp_buf);
        value = ((value << 8) & 0x0000FF00)| (*(temp_buf + 1));
        value = ((value << 8) & 0x00FFFF00)| (*(temp_buf + 2));
        value = ((value << 8) & 0xFFFFFF00)| (*(temp_buf + 3));
        metric_rsp->descriptorlist[j].link_metrics.rx_total_packet = value;
        temp_buf += 4;
        /*RSSI*/
        metric_rsp->descriptorlist[i].link_metrics.rssi = *(temp_buf);
        temp_buf += 1;
    }

    return (length+3);
}

int parse_link_metrics_result_code_type_tlv(unsigned char *buf)
{
    unsigned char *temp_buf;
    int length = 0;

    temp_buf = buf;
    /*shift to length field*/
    temp_buf++;

    /*calculate tlv length*/
    length = (*temp_buf);
    length = (length << 8) & 0xFF00;
    length = length |(*(temp_buf+1));
    /*shift to tlv value field*/
    temp_buf+=2;

    /*get result of invalid neighbor*/
    if((*temp_buf) == 0)
        return -1;

    return (length + 3);
}
#endif

int parse_push_button_event_notification_tlv(unsigned char *buf,
            iee802_11_info *wifi_info)
{
    unsigned char *temp_buf;
    int length = 0, i = 0;
    unsigned char num_media = 0;
    unsigned short media_type;
    unsigned char media_info_len = 0;

    temp_buf = buf;
    /*shift to length field*/
    temp_buf++;

    /*calculate tlv length*/
    length = (*temp_buf);
    length = (length << 8) & 0xFF00;
    length = length |(*(temp_buf+1));
    /*shift to tlv value field*/
    temp_buf+=2;

    num_media = *temp_buf;
    temp_buf++;

    if(num_media > 0)
    {
        for(i=0;i<num_media;i++)
        {
            media_type = *temp_buf;
            media_type = ((media_type << 8) & 0xFF00)| (*(temp_buf + 1));
            temp_buf += 2;

            debug(DEBUG_TRACE, "media_type in pb event notify = 0x%x\n",media_type);
            media_info_len = *temp_buf;
            temp_buf += 1;
            /* AP is a registrar, and the received push button event notification
             * carried the wifi media information, do not launch PBC sequece
             * on wifi AP
             */
            if((media_type >= IEEE802_11_GROUP) && (media_type < IEEE1901_GROUP))
            {
                wifi_info->no_need_start_pbc = 1;
            }
            temp_buf += media_info_len;
        }
    }

    return (length + 3);
}

int parse_search_role_tlv(
        unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == SEARCH_ROLE_TLV_TYPE)
        temp_buf++;
    else
    {
        return -1;
    }

    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    temp_buf+=2;

    if(length != SEARCH_ROLE_LENGTH)
        return -1;

    if((*temp_buf) != ROLE_REGISTRAR)
        return -1;

    return (length+3);
}

int parse_auto_config_freq_band_tlv(
        unsigned char *buf, unsigned char *band)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;
    if((*temp_buf) == AUTO_CONFIG_FREQ_BAND_TLV_TYPE)
        temp_buf++;
    else
    {
        return -1;
    }
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    temp_buf+=2;

    if(length != AUTOCONFIG_FREQ_BAND_LENGTH)
        return -1;

	*band = *temp_buf;

    return (length+3);
}

int parse_supported_role_tlv(
        unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == SUPPORT_ROLE_TLV_TYPE)
        temp_buf++;
    else
    {
        return -1;
    }

    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    temp_buf+=2;

    if(length != SUPPORTED_ROLE_LENGTH)
        return -1;
    if((*temp_buf) != ROLE_REGISTRAR)
        return -1;
	debug(DEBUG_TRACE, "support ROLE_REGISTRAR\n");
    return (length+3);
}

int parse_supported_freq_band_tlv(
        unsigned char *buf, unsigned char *band)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == SUPPORT_FREQ_BAND_TLV_TYPE)
        temp_buf++;
    else
        return -1;

    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    temp_buf+=2;

    if(length != SUPPORTED_FREQ_BAND_LENGTH)
        return -1;
	*band = *temp_buf;

	debug(DEBUG_TRACE, "support band %s\n", (*band) ? "5g":"2g");
    return (length+3);
}

int parse_wsc_tlv(
        unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == WSC_TLV_TYPE)
        temp_buf++;
    else
        return -1;

    /*calculate tlv length*/
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    /*shift to tlv value field*/
    temp_buf+=2;

    if(wsc_attr_success != parse_wsc_msg(ctx, temp_buf, length))
        return -1;

    return (length+3);
}
