/*
 * cleopatre/application/p1905_managerd/src/cmdu.c
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <errno.h>
#include <linux/if_packet.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <syslog.h>
#include <assert.h>
#include <linux/if_ether.h>
#include <asm/byteorder.h>
#include <fcntl.h>
#include <unistd.h>

#include "p1905_managerd.h"
#include "topology.h"
#include "cmdu.h"
#include "cmdu_retry_message.h"
#include "message_wait_queue.h"
#include "byteorder.h"
#include "debug.h"
#include "topology.h"
#include "ethernet_layer.h"
#include "mapfilter_if.h"
#include "common.h"

extern int debug_level;
extern int find_al_mac_address(struct p1905_managerd_ctx *ctx,
	unsigned char *mac_addr, unsigned char *al_mac_addr);
extern int makeAddr(const char *name, struct sockaddr_un *pAddr,
	socklen_t *pSockLen);
extern int set_opt_not_forward_dest(unsigned char *inputmac);
extern void init_fragment_queue();
extern void uninit_fragment_queue();
extern int _1905_write_mid(char *name, unsigned short mid);
//extern int get_cmdu_tlv_length(unsigned char *buf);






TAILQ_HEAD(list_head_txq, txq_list) cmdu_txq_head;
extern unsigned char p1905_multicast_address[6];

void delete_cmdu_txq_all(void);
int cmdu_send(struct p1905_managerd_ctx *ctx, unsigned char *dmac,
                     unsigned char *smac, msgtype mtype, unsigned short mid,
                     unsigned char *buffer, int if_index);
int cmdu_bridge_relay(struct p1905_managerd_ctx *ctx,
                             unsigned char *buf, int len, int if_index);
int cmdu_bridge_relay_send(struct p1905_managerd_ctx *ctx,
            unsigned char *smac, unsigned char *buf, int len);




/**
 * check destination mac address of received CMDU. if this CMDU is not for us
 * , we need to ignore this CMDU
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \param  da   pointer of destination MAC.
 * \return  1: for us 0: not for us.
 */
 unsigned char cmdu_dst_mac_check(struct p1905_managerd_ctx *ctx,
        unsigned char *da)
{
    int i = 0;

    if((!memcmp(da, ctx->p1905_al_mac_addr, ETH_ALEN)) ||
        (!memcmp(da, p1905_multicast_address, ETH_ALEN)))
        return 1;

    for(i=0; i<ITF_NUM; i++) {
        if(!memcmp(da, ctx->itf[i].mac_addr, ETH_ALEN))
            return 1;
    }

	return 0;
}


/**
 * check source mac address of received CMDU. if this received CMDU is send by us, drop it
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \param  da   pointer of source MAC.
 * \return  1: from us 0: not from us.
 */
unsigned char cmdu_src_mac_check(struct p1905_managerd_ctx *ctx,
        unsigned char *sa)
{
    int i= 0;

    if((!memcmp(sa, ctx->p1905_al_mac_addr, ETH_ALEN)))
        return 1;
	for(i = 0; i < ctx->itf_number; i++)
	{
		if(!memcmp(sa, ctx->itf[i].mac_addr, ETH_ALEN))
			return 1;
	}

    return 0;
}

/**
 *  bridge init
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \return  error code.
 */
int bridge_init(struct p1905_managerd_ctx *ctx)
{
    int f, s;
    struct ifreq ifr;

    if ((f=socket(PF_PACKET, SOCK_RAW, cpu2be16(ETH_P_ALL)))<0)
        return -1;

    strcpy(ifr.ifr_name, (char *)ctx->br_name);
    if ((s = ioctl(f, SIOCGIFFLAGS, &ifr))<0){
      close(f);
      return-1;
    }

    ifr.ifr_flags |= IFF_PROMISC;
    if ((s = ioctl(f, SIOCSIFFLAGS, &ifr)) < 0){
		close(f);
		return -1;
    }
    debug(DEBUG_OFF, "Setting interface ::: %s ::: to promisc\n", ifr.ifr_name);
	close(f);

    return 0;
}

/**
 * init cmdu tx queue.
 *
 */
void cmdu_txq_init(void)
{
    TAILQ_INIT(&cmdu_txq_head);
}

int concurrent_sock_init(struct p1905_managerd_ctx *ctx)
{
	socklen_t server_len;
	int status = 0;
	struct sockaddr_un sock_addr;

	ctx->sock_inf_recv_1905[FIRST_VITUAL_ITF] = socket(AF_UNIX, SOCK_STREAM , 0);
	if(ctx->sock_inf_recv_1905[FIRST_VITUAL_ITF] < 0)
	{
		debug(DEBUG_ERROR, "open sock_concurrent fail\n");
		return -1;
	}

	if (ctx->role == CONTROLLER)
		makeAddr("map_controller", &sock_addr, &server_len);
	else
		makeAddr("map_agent", &sock_addr, &server_len);

	status = bind(ctx->sock_inf_recv_1905[FIRST_VITUAL_ITF], (struct sockaddr *)&sock_addr, server_len);
	if(status < 0)
	{
		unlink(sock_addr.sun_path);
		close(ctx->sock_inf_recv_1905[FIRST_VITUAL_ITF]);
		debug(DEBUG_ERROR, "bind sock_concurrent fail(%s)\n", strerror(errno));
		return -1;
	}

	if(listen(ctx->sock_inf_recv_1905[FIRST_VITUAL_ITF], 1) < 0)
	{
		unlink(sock_addr.sun_path);
		close(ctx->sock_inf_recv_1905[FIRST_VITUAL_ITF]);
		debug(DEBUG_ERROR, "listen sock_concurrent fail(%s)\n", strerror(errno));
		return -1;
	}

	return 0;
}

int concurrent_sock_send(struct p1905_managerd_ctx *ctx, char *buffer_send, int length)
{
	/*the buf content should be the right wapp command*/
	struct sockaddr_un sock_addr;
	int fd, status, len;
	//int flags;
	socklen_t client_len;

	len = -1;

	fd = socket(AF_UNIX, SOCK_STREAM , 0);
	if(fd < 0)
	{
		debug(DEBUG_ERROR, "open socket error(%s)\n", strerror(errno));
		goto end;
	}

/*
	change to nonblock way to connect peer socket
*/
	//flags = fcntl(fd,F_GETFL,0);
	//fcntl(fd,F_SETFL,flags|O_NONBLOCK);

	if (ctx->role == CONTROLLER) {
		makeAddr("map_agent", &sock_addr, &client_len);
	} else {
		makeAddr("map_controller", &sock_addr, &client_len);
	}

	status = connect(fd, (struct sockaddr *)&sock_addr, client_len);

	if (status < 0) {
		if (ctx->role == CONTROLLER) {
			debug(DEBUG_TRACE, "connect map_agent error(%s)", strerror(errno));
		} else {
			debug(DEBUG_TRACE, "connect map_controller error(%s)", strerror(errno));
		}
		goto error1;
	}


	len = send(fd, buffer_send, length, 0);
	if(len < 0)
	{
		if (ctx->role == CONTROLLER) {
			debug(DEBUG_TRACE, "send to map_agent error(%s)", strerror(errno));
		} else {
			debug(DEBUG_TRACE, "send to map_controller error(%s)", strerror(errno));
		}
		goto error1;
	}

error1:
	close(fd);
end:
	return len;

}

void concurrent_sock_deinit(struct p1905_managerd_ctx* ctx)
{
	if (ctx->role == CONTROLLER)
		unlink("map_controller");
	else
		unlink("map_agent");
	close(ctx->sock_inf_recv_1905[FIRST_VITUAL_ITF]);
}

#ifdef SUPPORT_CONTROL_SOCKET
int _1905_ctrl_sock_init(struct p1905_managerd_ctx *ctx)
{
	struct sockaddr_un addr;
	socklen_t server_len;

	/* Initialize control interface */
	ctx->ctrl_sock = socket(AF_UNIX, SOCK_STREAM, 0);
	if (ctx->ctrl_sock < 0) {
		debug(DEBUG_ERROR, "open ctrl_sock fail\n");
		return -1;
	}

	if (ctx->role == CONTROLLER)
		makeAddr("/tmp/1905ctrl_controller", &addr, &server_len);
	else
		makeAddr("/tmp/1905ctrl_agent", &addr, &server_len);

	if (bind(ctx->ctrl_sock, (struct sockaddr *)&addr, server_len) < 0) {
		debug(DEBUG_ERROR, "bind addr to ctrl_sock fail(%s)\n", strerror(errno));
		unlink(addr.sun_path);
		close(ctx->ctrl_sock);
		return -1;
	}

	if(listen(ctx->ctrl_sock, 1) < 0)
	{
		unlink(addr.sun_path);
		close(ctx->ctrl_sock);
		debug(DEBUG_ERROR, "listen ctrl_sock fail(%s)\n", strerror(errno));
		return -1;
	}

	return 0;
}

void _1905_ctrl_interface_recv_and_parse(struct p1905_managerd_ctx *ctx,
		char *buf, int len)
{
	struct sockaddr_un from;
	int lenfrom = sizeof(from);
	int recv_fd = 0;
	char *file = NULL;

	recv_fd = accept(ctx->ctrl_sock, (struct sockaddr *)&from, (socklen_t *)&lenfrom);
	if (recv_fd < 0) {
		debug(DEBUG_ERROR, "accept ctrl_sock fail\n");
		return;
	}

	len = recv(recv_fd, buf, len, 0);
	if (0 >= len) {
		debug(DEBUG_ERROR, "recv ctrl_sock fail\n");
		close(recv_fd);
		return;
	}
	close(recv_fd);

	buf[len] = '\0';

	if (os_strncmp(buf, "dev_send_1905", os_strlen("dev_send_1905")) == 0) {
		file = buf + sizeof("dev_send_1905") + 1;
		send_1905_raw_data(ctx, file);
	} else if (os_strncmp(buf, "dev_set_config", os_strlen("dev_set_config")) == 0) {
		if (ctx->role == CONTROLLER) {
			file = buf + sizeof("dev_set_config") + 1;
			read_1905_bss_config(ctx, file);
		} else {
			debug(DEBUG_ERROR, "agent mode, no need hanle dev_set_config\n");
		}
	} else if (os_strncmp(buf, "dump_topology_info", os_strlen("dump_topology_info")) == 0) {
		dump_topology_info(ctx);
	} else if (os_strncmp(buf, "dump_topology_tree", os_strlen("dump_topology_tree")) == 0) {
		dump_topology_tree(ctx);
		eth_layer_port_clients_dump();
	} else if (os_strncmp(buf, "log_level", os_strlen("log_level")) == 0) {
		int level = 0;
		char *p = buf + os_strlen("log_level") + 1;

		level = atoi(p);
		set_debug_level(level);
	} else if (os_strncmp(buf, "get_eth_port_type", os_strlen("get_eth_port_type")) == 0) {
		test_eth_layer_get_port_type();
	}  else if (os_strncmp(buf, "get_eth_entry", os_strlen("get_eth_entry")) == 0) {
		test_eth_layer_get_client_entry();
	}  else if (os_strncmp(buf, "find_eth_entry", os_strlen("find_eth_entry")) == 0) {
		unsigned char mac[ETH_ALEN];
		if(!hwaddr_aton(buf + os_strlen("find_eth_entry") + 1, mac))
			test_eth_layer_search_table_entry(mac);
	} else if (os_strncmp(buf, "get_eth_port_status", os_strlen("get_eth_port_status")) == 0) {
		int port_index = 0;
		char *p = buf + os_strlen("get_eth_port_status") + 1;
		port_index = atoi(p);
		test_eth_layer_get_port_status(port_index);
	} else if (os_strncmp(buf, "mapfilter_debug", os_strlen("mapfilter_debug")) == 0) {
		mapfilter_dump_debug_info();
	} else if (os_strncmp(buf, "mapfilter_set_primary", os_strlen("mapfilter_set_primary")) == 0) {
		unsigned char mac[ETH_ALEN];

		debug(DEBUG_ERROR, "%s\n", buf);
		if(!hwaddr_aton(buf + os_strlen("mapfilter_set_primary") + 1, mac)) {
			struct local_interface itf;
			int primary = INF_UNKNOWN;
			char *p = NULL;

			os_memcpy(itf.mac, mac, ETH_ALEN);
			itf.dev_type = APCLI;
			p = buf + os_strlen("mapfilter_set_primary") + 1 + 18;
			primary = atoi(p);

			debug(DEBUG_ERROR, "mapfilter_set_primary %02x:%02x:%02x:%02x:%02x:%02x %02x\n", PRINT_MAC(mac), primary);
			mapfilter_set_primary_interface(&itf, primary);
		}
	} else if (os_strncmp(buf, "mapfilter_set_up_path", os_strlen("mapfilter_set_up_path")) == 0) {
		unsigned char mac[ETH_ALEN], mac2[ETH_ALEN];

		debug(DEBUG_ERROR, "%s\n", buf);
		if(!hwaddr_aton(buf + os_strlen("mapfilter_set_up_path") + 1, mac)) {
			struct local_interface in, out;
			char *p = NULL;

			os_memcpy(in.mac, mac, ETH_ALEN);

			p = buf + os_strlen("mapfilter_set_up_path") + 1 + 18;
			if (!hwaddr_aton(p, mac2)) {
				os_memcpy(out.mac, mac2, ETH_ALEN);
				debug(DEBUG_ERROR, "mapfilter_set_up_path %02x:%02x:%02x:%02x:%02x:%02x->%02x:%02x:%02x:%02x:%02x:%02x\n",
					PRINT_MAC(mac), PRINT_MAC(mac2));
				mapfilter_set_uplink_path(&in, &out);
			}
		}
	} else if (os_strncmp(buf, "show_PON_dev", os_strlen("show_PON_dev")) == 0) {
		debug(DEBUG_ERROR, "%s\n", buf);
		show_PON_dev(ctx);
	} else if (strncmp(buf, "set_debug_level", strlen("set_debug_level")) == 0) {
		debug(DEBUG_OFF, "buf %s\n", buf);
		debug_level = atoi(buf + strlen("set_debug_level") + 1);
		debug(DEBUG_OFF, "change debug level to %d\n", debug_level);
	} else if (strncmp(buf, "set_ts_config", strlen("set_ts_config")) == 0){
#ifdef MAP_R2
		char *tmp = buf + os_strlen("set_ts_config") + 1;
		if (strncmp(tmp, "pvlan", strlen("pvlan")) == 0) {
			cmd_set_ts_pvlan(ctx, tmp + os_strlen("pvlan") + 1);
		} else if (strncmp(tmp, "policy", strlen("policy")) == 0) {
			tmp = tmp + os_strlen("policy") + 1;
			if (strncmp(tmp, "clean", strlen("clean")) == 0)
				cmd_set_ts_policy_clear(ctx);
			else if (strncmp(tmp, "done", strlen("done")) == 0)
				cmd_set_ts_policy_done(ctx);
			else
				cmd_set_ts_policy(ctx, tmp);
		} else {
			debug(DEBUG_ERROR, "%s no such command\n", __func__);
		}
#else
	debug(DEBUG_ERROR, "%s set_ts_config only for R2 \n", __func__);
#endif// #ifdef MAP_R2
	}else {
		debug(DEBUG_ERROR, "%s no such command\n", __func__);
	}
}

void  _1905_ctrl_sock_deinit(struct p1905_managerd_ctx *ctx)
{
	close(ctx->ctrl_sock);
	debug(DEBUG_OFF, "\n");
}
#endif

/**
 * Initialize all parameters for CMDU.
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \return  error code.
 */
int cmdu_init(struct p1905_managerd_ctx *ctx)
{
	struct sockaddr_ll sll;
	struct ifreq ifr;
	int i = 0;
	int nSndBufLen = 0,nRcvBufLen = 0;
	socklen_t optlen = sizeof(int);

	ctx->mid = 0;
	ctx->need_relay = 0;

	if(0 > (ctx->sock_br = socket(AF_PACKET, SOCK_RAW, ETH_P_1905)))
	{
	    debug(DEBUG_ERROR, "cannot open socket on %s (%s)", ctx->br_name,
	           strerror(errno));
	    return -1;
	}

	strncpy(ifr.ifr_name, (char *)ctx->br_name, IFNAMSIZ);
	if(-1 == (ioctl(ctx->sock_br, SIOCGIFINDEX, &ifr)))
	{
	    debug(DEBUG_ERROR, "cannot get interface %s index (%s)\n", ctx->br_name,
	           strerror(errno));
	    close(ctx->sock_br);
	    return -1;
	}

	if (getsockopt(ctx->sock_br, SOL_SOCKET, SO_SNDBUF, (void *)&nSndBufLen, &optlen) < 0) {
		nSndBufLen = 1024*1024;
	}
	else {
		nSndBufLen = nSndBufLen * 2;
	}

	if (getsockopt(ctx->sock_br, SOL_SOCKET, SO_RCVBUF, (void *)&nRcvBufLen, &optlen) < 0) {
		nRcvBufLen = 1024*1024;
	}
	else {
		nRcvBufLen = nRcvBufLen * 2;
	}

	if (setsockopt(ctx->sock_br, SOL_SOCKET, SO_SNDBUF, (const char*)&nSndBufLen, sizeof(int)) < 0) {
		printf("warning: %s set send buffer size failed, %s\n", __func__, strerror(errno));
	} else {
		printf("%s set send buffer size %d\n", __func__, nSndBufLen);
	}

	if (setsockopt(ctx->sock_br, SOL_SOCKET, SO_RCVBUF, (const char*)&nRcvBufLen, sizeof(int)) < 0) {
		printf("warning: %s set recv buffer size failed, %s\n", __func__, strerror(errno));
	} else {
		printf("%s set recv buffer size %d\n", __func__, nRcvBufLen);
	}

	/*Create a receive connection of CMDU on bridge interface*/
	sll.sll_family = AF_PACKET;
	sll.sll_ifindex = ifr.ifr_ifindex;
	sll.sll_protocol = cpu2be16(ETH_P_1905);
	if(-1 == (bind(ctx->sock_br, (struct sockaddr *)&sll,
	          sizeof(struct sockaddr_ll))))
	{
	    debug(DEBUG_ERROR, "cannot bind raw socket to interface %s (%s)",
	           ctx->br_name, strerror(errno));
	    close(ctx->sock_br);
	    return -1;
	}

	/*create a connection to recevie/send CMDU on each interface*/
	for(i = FIRST_VITUAL_ITF + 1; i < ctx->itf_number; i++)
	{
	    if(0 > (ctx->sock_inf_recv_1905[i] = socket(AF_PACKET, SOCK_RAW, ETH_P_1905)))
	    {
	        debug(DEBUG_ERROR, "cannot open socket on %s (%s)", ctx->itf[i].if_name,
				strerror(errno));
			close(ctx->sock_br);
	        return -1;
	    }

	    strncpy(ifr.ifr_name, (char*)ctx->itf[i].if_name, IFNAMSIZ);
	    if(-1 == (ioctl(ctx->sock_inf_recv_1905[i], SIOCGIFINDEX, &ifr)))
	    {
	        debug(DEBUG_ERROR, "cannot get interface %s index (%s) i=%d\n",
				ctx->itf[i].if_name, strerror(errno), i);
	        close(ctx->sock_inf_recv_1905[i]);
		 	close(ctx->sock_br);
	        return -1;
	    }

	    sll.sll_family = AF_PACKET;
	    sll.sll_ifindex = ifr.ifr_ifindex;
	    sll.sll_protocol = cpu2be16(ETH_P_1905);

	    /*Get br0 mac address*/
	    if(-1 == (ioctl(ctx->sock_inf_recv_1905[i], SIOCGIFHWADDR, &ifr)))
	    {
	        debug(DEBUG_ERROR, "cannot get interface %s mac address (%s)",
	               ctx->itf[i].if_name, strerror(errno));
	        close(ctx->sock_inf_recv_1905[i]);
			close(ctx->sock_br);
	        return -1;
	    }
	    memcpy(ctx->itf[i].mac_addr, ifr.ifr_hwaddr.sa_data, ETH_ALEN);

	    if(-1 == (bind(ctx->sock_inf_recv_1905[i], (struct sockaddr *)&sll,
	              sizeof(struct sockaddr_ll))))
	    {
	        debug(DEBUG_ERROR, "cannot bind raw socket to interface %s (%s)",
	               ctx->itf[i].if_name, strerror(errno));
	        close(ctx->sock_inf_recv_1905[i]);
			close(ctx->sock_br);
	        return -1;
	    }
	}

    /* Let bridge do not forward if dest address is local AL MAC address.
     * It is for 1905.1 unicast message usage.
     */
	if (set_opt_not_forward_dest(ctx->p1905_al_mac_addr) < 0) {
		close(ctx->sock_br);
		return -1;
	}


    /*initialize the cmdu transmit queue */
    cmdu_txq_init();

    /* initialize the cmdu fragment queue. It will queue fragment cmdu until
     * all fragmenets received
     */
    init_fragment_queue();

	/*
	 * initialization retry queue. it will queue message which needs receive 1905_ack until
	 * 1905_ack received or timeout
	*/
	 init_retry_queue();

	/*
	 * initialization message wait queue. it will queue unsolicited event while waiting for
	 * some specific event.
	*/
	init_message_wait_queue();

    return 0;
}

/**
 * uninit cmdu, it will be called when exit this process.
 *
 * \param  ctx  p1905_managerd_ctx context.
 */
void cmdu_uninit(struct p1905_managerd_ctx *ctx)
{
	int i = 0;

	close(ctx->sock_br);
	for(i = FIRST_VITUAL_ITF + 1; i < ctx->itf_number; i++)
	{
		close(ctx->sock_inf_recv_1905[i]);
	}
    /*delete all message queue in TX queue*/

	concurrent_sock_deinit(ctx);
#ifdef SUPPORT_CONTROL_SOCKET
	_1905_ctrl_sock_deinit(ctx);
#endif
    delete_cmdu_txq_all();

    /*delete all fragment message queue in fragment queue*/

    uninit_fragment_queue();
    /*delete all retry message in retry message queue*/
	uninit_retry_queue();

	uninit_message_wait_queue();
}

/**
 *  for cmdu message sniffer.
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \param  buf  pointer of buffer for tx.
 * \param  len  transmit length.
 * \return error code
 */
int cmdu_eth0_send(struct p1905_managerd_ctx *ctx,
                             unsigned char *buf, int len)
{
    int length;

    length = send(ctx->sock_eth0, buf, len, 0);

    if(0 > length)
    {
        debug(DEBUG_WARN, "relay failed on %s (%s)", ETH0_IFNAME,
               strerror(errno));
        return -1;
    }

    return 0;
}

int sniffer_init(struct p1905_managerd_ctx *ctx)
{
#if 0
    struct ifreq ifr;

    /*disable sniffer function when initailization*/
    ctx->sniffer_enable = 0;

    if(0 > (ctx->sock_eth0 = socket(AF_PACKET, SOCK_RAW, ETH_P_1905)))
    {
        syslog(LOG_WARNING, "cannot open socket on %s (%s)", ETH0_IFNAME,
               strerror(errno));
        return -1;
    }

    strncpy(ifr.ifr_name, (char*)ETH0_IFNAME, IFNAMSIZ);

    if(-1 == (ioctl(ctx->sock_eth0, SIOCGIFINDEX, &ifr)))
    {
	    syslog(LOG_WARNING, "cannot get interface %s index (%s)", ETH0_IFNAME,
               strerror(errno));
	    close(ctx->sock_eth0);
	    return -1;
    }

    ctx->eth0_sll.sll_family = AF_PACKET;
    ctx->eth0_sll.sll_ifindex = ifr.ifr_ifindex;
    ctx->eth0_sll.sll_protocol = htons(ETH_P_1905);

    /*Bind eth0 socket to this interface*/
    if(-1 == (bind(ctx->sock_eth0, (struct sockaddr *)&ctx->eth0_sll,
          sizeof(struct sockaddr_ll))))
    {
        syslog(LOG_WARNING, "cannot bind raw socket to interface %s (%s)",
            ETH0_IFNAME, strerror(errno));
        close(ctx->sock_eth0);
        return -1;
    }
#endif

	return 0;
}

void sniffer_uninit(struct p1905_managerd_ctx *ctx)
{
#if 0
    close(ctx->sock_eth0);
#endif
}

/**
 *  get the cmdu message in tx queue and send to bridge.
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \param  buffer  tx buffer.
 * \return  error code.
 */
int process_cmdu_txq(struct p1905_managerd_ctx *ctx, unsigned char *buffer)
{
    struct txq_list *tlist;
    int result = 0;
	int i = 0;

    if(TAILQ_EMPTY(&cmdu_txq_head))
    {
       debug(DEBUG_INFO, "txq empty\n");
       return 0;
    }

    TAILQ_FOREACH(tlist, &cmdu_txq_head, cmdu_txq_entry)
    {
        debug(DEBUG_TRACE, "txq send:\n");
        debug(DEBUG_TRACE, "dst %02x:%02x:%02x:%02x:%02x:%02x\n", PRINT_MAC(tlist->dmac));
        debug(DEBUG_TRACE, "src %02x:%02x:%02x:%02x:%02x:%02x\n", PRINT_MAC(tlist->smac));
        debug(DEBUG_TRACE, "mtype 0x%04x\n", tlist->mtype);
        debug(DEBUG_TRACE, "mid 0x%04x\n",tlist->mid);

		for(i = FIRST_VITUAL_ITF; i < ctx->itf_number; i++)
		{
			if(!memcmp(tlist->ifname, ctx->itf[i].if_name, strlen((char *)tlist->ifname)))
			{
				break;
			}
		}
		if(i >= ctx->itf_number)
		{
			debug(DEBUG_TRACE, "txq unkonwn inf name %s, tx by br\n", tlist->ifname);
			debug(DEBUG_TRACE, "mtype 0x%04x\n", tlist->mtype);
			i = -1;  //means tx by br
		} else {
			/*drop this packet that does not allow to send multicast*/
			if (!(ctx->itf[i].trx_config & TX_MUL) && !os_memcmp(tlist->dmac, p1905_multicast_address, ETH_ALEN)) {
				debug(DEBUG_TRACE, "drop send multicast packet on %s\n", ctx->itf[i].if_name);
				continue;
			}

			if (!(ctx->itf[i].trx_config & TX_UNI) && os_memcmp(tlist->dmac, p1905_multicast_address, ETH_ALEN)) {
				debug(DEBUG_TRACE, "drop send unicast packet on %s\n", ctx->itf[i].if_name);
				continue;
			}
		}

		/*map agent and controlller concurrent not set, so do not send any cmdu by virtual interface*/
		if(i == FIRST_VITUAL_ITF && ctx->concurrent != 1)
		{
			continue;
		}

        result = cmdu_send(ctx, tlist->dmac, tlist->smac,
                 tlist->mtype, tlist->mid, buffer, i);
        if(0 > result)
        {
           debug(DEBUG_ERROR, "txq send fail\n");
           return -1;
        }
		if(tlist->need_update_mid)
		{
			if(_1905_write_mid("/tmp/msg_id.txt", tlist->mid) < 0)
			{
				debug(DEBUG_ERROR, "update mid=0x%04x to dev_send_1905_mid.txt fail\n", tlist->mid);
			}
			else
			{
				debug(DEBUG_TRACE, "update mid=0x%04x to dev_send_1905_mid.txt success\n", tlist->mid);
			}
		}
    }
    delete_cmdu_txq_all();
	if (ctx->sta_notifier)
		ctx->sta_notifier = 0;
    return 0;
}

/**
 *  insert cmdu message into tx queue.
 *
 * \param  dmac  destination mac address.
 * \param  smac  source mac addr, it awii affect the forwarding of bridge.
 * \param  mtype  cmdu message type, defined in 1905.1 spec.
 * \param  mid  message id, defined in 1905.1 spec
 */
void insert_cmdu_txq(unsigned char *dmac, unsigned char *smac, msgtype mtype,
                    unsigned short mid, unsigned char* ifname, unsigned char need_update_mid)
{
    struct txq_list *tlist;

    tlist = (struct txq_list*)malloc(sizeof(struct txq_list));
    if (!tlist)
        return;
    memset(tlist, 0, sizeof(struct txq_list));
    memcpy(tlist->dmac, dmac, 6);
    memcpy(tlist->smac, smac, 6);
	strncpy((char *)tlist->ifname, (char *)ifname, strlen((char *)ifname));
    tlist->mtype = mtype;
    tlist->mid = mid;
	tlist->need_update_mid = need_update_mid;
    TAILQ_INSERT_TAIL(&cmdu_txq_head, tlist, cmdu_txq_entry);
}

/**
 *  delete all cmdu message in tx queue.
 *
 */
void delete_cmdu_txq_all(void)
{
    struct txq_list *tlist, *tlist_temp;

    tlist = TAILQ_FIRST(&cmdu_txq_head);
    while(tlist)
    {
        tlist_temp = TAILQ_NEXT(tlist, cmdu_txq_entry);
        TAILQ_REMOVE(&cmdu_txq_head, tlist, cmdu_txq_entry);
        free(tlist);
        tlist = tlist_temp;
    }
}

/**
 *  Fragment tx. Use this funciton when CMDU message size exceed max size.
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \param  tlv  pointer of tlv start position.
 * \param  msg_hdr  pointer of message header start position.
 * \param  buffer   pointer of start position of whole message.
 * \return error code
 */
int cmdu_tx_fragment(struct p1905_managerd_ctx *ctx,
	unsigned char *buffer, unsigned short buflen, int if_index)
{
    unsigned short offset = 0;
    unsigned short tmp_length = 0;
	cmdu_message_header *msg_hdr = NULL;
    unsigned char *tlv_buf_raw = NULL, *tlv_buf = NULL;
    unsigned char *temp_buf = NULL;
	unsigned char *tlv = NULL;
    unsigned short tlv_len = 0, total_tlv_len = 0;
    int len = 0, i = 0;

	msg_hdr = (cmdu_message_header *)(buffer + ETH_HLEN);
	tlv = buffer + ETH_HLEN + CMDU_HLEN;
	total_tlv_len = buflen - ETH_HLEN - CMDU_HLEN;
	tlv_buf = (unsigned char *)malloc(total_tlv_len);
	if (!tlv_buf) {
		debug(DEBUG_ERROR, "alloc tlv_buf fail\n");
		return -1;
	}
	tlv_buf_raw = tlv_buf;

	memcpy(tlv_buf, tlv, total_tlv_len);
	temp_buf = tlv_buf;

    for (;;) {
        if ((*temp_buf) == END_OF_TLV_TYPE) {
            /* if last fragment is larger than MAX_TLVS_LENGTH,
             * we need to seperate this fragment into two fragments
             * the total length of end of tlv is 3
             */
            if ((offset + 3) > MAX_TLVS_LENGTH) {
                memcpy(tlv, tlv_buf, offset);
                msg_hdr->fragment_id = i;
                msg_hdr->last_fragment_indicator = 0;
                len = offset + CMDU_HLEN + ETH_HLEN;
				if (if_index != FIRST_VITUAL_ITF)
                	len = send(if_index < 0 ? ctx->sock_br : ctx->sock_inf_recv_1905[if_index], buffer, len, 0);
				else
					len = concurrent_sock_send(ctx, (char *)buffer, len);
                if(0 > len)
                   goto error;

				debug(DEBUG_TRACE,"msgtype=0x%04x, interface_id=%d interface=%s fid=%d mid=%d lf=%d\n",
					be2cpu16(msg_hdr->message_type), if_index,
					(if_index < 0 ? ctx->br_name : ctx->itf[if_index].if_name),
					msg_hdr->fragment_id, be2cpu16(msg_hdr->message_id),
					msg_hdr->last_fragment_indicator);
                /*add for fragment id*/
                i++;

                memset(tlv, 0, MIN_TLVS_LENGTH);
                msg_hdr->fragment_id = i;
                msg_hdr->last_fragment_indicator = 1;

                len = MIN_TLVS_LENGTH + CMDU_HLEN + ETH_HLEN;
				if (if_index != FIRST_VITUAL_ITF)
                	len = send(if_index < 0 ? ctx->sock_br : ctx->sock_inf_recv_1905[if_index], buffer, len, 0);
				else
					len = concurrent_sock_send(ctx, (char *)buffer, len);
                if(0 > len)
                    goto error;

				debug(DEBUG_TRACE,"msgtype=0x%04x, interface_id=%d interface=%s fid=%d mid=%d lf=%d\n",
					be2cpu16(msg_hdr->message_type), if_index,
					(if_index < 0 ? ctx->br_name : ctx->itf[if_index].if_name),
					msg_hdr->fragment_id, be2cpu16(msg_hdr->message_id),
					msg_hdr->last_fragment_indicator);
            } else {
                /* It is the last fragment so we set last fragment ind to 1.
                * And send this buffer
                */
                msg_hdr->fragment_id = i;
                msg_hdr->last_fragment_indicator = 1;

                memcpy(tlv, tlv_buf, (offset + 3));

                if ((offset + 3) < MIN_TLVS_LENGTH) {
                    memset((tlv+(offset + 3)), 0, MIN_TLVS_LENGTH-(offset + 3));
                    tmp_length = MIN_TLVS_LENGTH;
                } else {
                    tmp_length = (offset + 3);
                }
                len = tmp_length + CMDU_HLEN + ETH_HLEN;

				if (if_index != FIRST_VITUAL_ITF)
                	len = send(if_index < 0 ? ctx->sock_br : ctx->sock_inf_recv_1905[if_index], buffer, len, 0);
				else
					len = concurrent_sock_send(ctx, (char *)buffer, len);
                if(0 > len)
                   goto error;
				debug(DEBUG_TRACE,"msgtype=0x%04x, interface_id=%d interface=%s fid=%d mid=%d lf=%d\n",
					be2cpu16(msg_hdr->message_type), if_index,
					(if_index < 0 ? ctx->br_name : ctx->itf[if_index].if_name),
					msg_hdr->fragment_id, be2cpu16(msg_hdr->message_id),
					msg_hdr->last_fragment_indicator);
            }
            goto out;
        }

        /* below is non-last fragment case.
         * We should go here first when fragment tx happen.
         */
        tlv_len = get_cmdu_tlv_length(temp_buf);
        if ((offset + tlv_len) > MAX_TLVS_LENGTH) {

            memcpy(tlv, tlv_buf, offset);

            /* set fragment id & last fragment ind in cmdu message header.
             * In this case, last fragment ind must be 0 because no end_of_tlv.
             */
            msg_hdr->fragment_id = i;
            msg_hdr->last_fragment_indicator = 0;

            len = offset + CMDU_HLEN + ETH_HLEN;
			if(if_index != FIRST_VITUAL_ITF)
            	len = send(if_index < 0 ? ctx->sock_br : ctx->sock_inf_recv_1905[if_index], buffer, len, 0);
			else
				len = concurrent_sock_send(ctx, (char *)buffer, len);
            if(0 > len)
                goto error;

			debug(DEBUG_TRACE,"msgtype=0x%04x, interface_id=%d interface=%s fid=%d mid=%d lf=%d\n",
				be2cpu16(msg_hdr->message_type), if_index,
				(if_index < 0 ? ctx->br_name : ctx->itf[if_index].if_name),
				msg_hdr->fragment_id, be2cpu16(msg_hdr->message_id),
				msg_hdr->last_fragment_indicator);
            /*shift to start position of not sent cmdu payload */
            tlv_buf += offset;

            /* record the tlv_len, which was gotten by get_cmdu_tlv_length
             * Don't worry about buffer address, because we will shift buffer
             * to correct position in the end of this for loop.
             */
            offset = tlv_len;

            /*for fragment id use*/
            i++;
        } else {
            offset += tlv_len;
        }

        /*shift buffer to correct position*/
        temp_buf += tlv_len;
    }

out:
	free(tlv_buf_raw);
	return 0;
error:
	debug(DEBUG_ERROR, "error msgtype=0x%04x, interface_id=%d interface=%s fid=%d mid=%d lf=%d errno(%d-%s)\n",
		be2cpu16(msg_hdr->message_type), if_index,
		(if_index < 0 ? ctx->br_name : ctx->itf[if_index].if_name),
		msg_hdr->fragment_id, be2cpu16(msg_hdr->message_id),
		msg_hdr->last_fragment_indicator, errno, strerror(errno));
	hex_dump_all("error send frag",buffer,len);
	free(tlv_buf_raw);
	return -1;
}

struct _config_sync {
	unsigned char al_mac[ETH_ALEN];
	unsigned char band;
};

void config_state_set_handler(void *context, void* parent_leaf, void *current_leaf, void *data)
{
	struct _config_sync *sync_data = (struct _config_sync*)data;
	unsigned char *al_mac = sync_data->al_mac;
	struct leaf *cur_leaf = (struct leaf *)current_leaf;

	if (!os_memcmp(cur_leaf->al_mac, al_mac, ETH_ALEN)) {
		cur_leaf->config_status |= sync_data->band;
	}
}

/**
 *  create & send cmdu message to the corresponding interface.
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \param  dmac  destination mac address.
 * \param  smac  source mac address.
 * \param  mtype   cmdu message type.
 * \param  mid   cmdu message id.
 * \param  buffer   cmdu buffer for sending.
 * \return error code
 */
int cmdu_send(struct p1905_managerd_ctx *ctx, unsigned char *dmac,
                     unsigned char *smac, msgtype mtype, unsigned short mid,
                     unsigned char *buffer, int if_index)
{
    struct ethhdr *eth_hdr;
    cmdu_message_header *msg_hdr;
    unsigned char *msg;
    unsigned short msg_leng = 0;
    unsigned short len;
    unsigned char *tlv;
	struct agent_list_db *agent = NULL;

    eth_hdr = (struct ethhdr*)buffer;
    memcpy(eth_hdr->h_dest, dmac, ETH_ALEN);
    memcpy(eth_hdr->h_source, smac, ETH_ALEN);
    eth_hdr->h_proto = cpu2be16(ETH_P_1905);

    /*start position of message header*/
    msg = buffer + ETH_HLEN;
    msg_hdr = (cmdu_message_header*)(msg);

    /*start position of tlvs */
    tlv = msg + CMDU_HLEN;

    switch(mtype)
    {
            case e_topology_discovery:
                msg_leng = create_topology_discovery_message(msg,
                           ctx->p1905_al_mac_addr,ctx->itf[if_index].mac_addr, 0, NULL);
                break;
			case e_topology_discovery_with_vendor_ie:
				{
					unsigned char vs_info[16] = {0};
					unsigned short vs_len = 0;

					vs_len = create_vs_info_for_specific_discovery(vs_info);

					msg_leng = create_topology_discovery_message(msg,
                           ctx->p1905_al_mac_addr,ctx->itf[if_index].mac_addr, vs_len, vs_info);
					if (msg_leng == 0)
						return 0;
				}
				break;
			case e_vendor_specific_topology_discovery:
				{
					unsigned char vs_info[16] = {0};
					unsigned short vs_len = 0;

					vs_len = create_vs_info_for_specific_discovery(vs_info);

	                msg_leng = create_vendor_specific_topology_discovery_message(msg,
	                           ctx->p1905_al_mac_addr,ctx->itf[if_index].mac_addr, vs_len, vs_info);
					if (msg_leng == 0)
						return 0;
				}
                break;
            case e_topology_notification:
                msg_leng = create_topology_notification_message(msg,
                           ctx->p1905_al_mac_addr, &ctx->sinfo.sassoc_evt,
                           ctx->sta_notifier, 0, NULL);
                break;
			case e_topology_notification_with_vendor_ie:
				{
					unsigned char vs_info[32] = {0};
					unsigned short vs_len = 0;

					vs_len = create_vs_info_for_specific_notification(ctx, vs_info);
					msg_leng = create_topology_notification_message(msg,
						ctx->p1905_al_mac_addr, &ctx->sinfo.sassoc_evt,
						ctx->sta_notifier, vs_len, vs_info);
				}
                break;
            case e_topology_query:
                msg_leng = create_topology_query_message(msg
#ifdef MAP_R2
				, ctx->map_version
#endif
				);
                break;
            case e_topology_response:
                msg_leng = create_topology_response_message(msg,
                           ctx->p1905_al_mac_addr, ctx, ctx->br_cap,
                           ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
                           &(ctx->topology_entry.tprdb_head), ctx->service,
                           &(ctx->ap_cap_entry.oper_bss_head),
                           &(ctx->ap_cap_entry.assoc_clients_head), ctx->cnt);
                break;
            case e_vendor_specific:
                msg_leng = create_vendor_specific_message(msg, ctx);
                break;
            case e_link_metric_query:
                msg_leng = create_link_metrics_query_message(msg,
                           ctx->link_metric_query_para.target,
                           ctx->link_metric_query_para.type);
                break;
            case e_link_metric_response:
                msg_leng = create_link_metrics_response_message(msg,
                            ctx->link_metric_response_para.target,
                            ctx->link_metric_response_para.type,
                            ctx->p1905_al_mac_addr,
                            ctx->itf, ctx->p1905_neighbor_dev,
                            &(ctx->topology_entry.tpddb_head), ctx);
                break;
            case e_ap_autoconfiguration_search:
                msg_leng = ap_autoconfiguration_search_message(msg, ctx);
                break;
            case e_ap_autoconfiguration_wsc_m1:
                msg_leng = ap_autoconfiguration_wsc_message(msg, ctx, dmac, MESSAGE_TYPE_M1);
                break;
            case e_ap_autoconfiguration_response:
                msg_leng = ap_autoconfiguration_response_message(msg, ctx);
                break;
            case e_ap_autoconfiguration_wsc_m2:
				/*set the state of the leaf to CONFIGURED*/
				{
					struct _config_sync sync_data;

					find_agent_info(ctx, dmac, &agent);
					if (agent) {
						agent->band_cap |= ctx->band_cap;
						agent->band_configured |= ctx->peer_bss_rf_band;
						debug(DEBUG_WARN, "agent(%02x:%02x:%02x:%02x:%02x:%02x) band_cap=%02x\n",
							PRINT_MAC(agent->almac), agent->band_cap);
					}

					os_memcpy(sync_data.al_mac, dmac, ETH_ALEN);
					sync_data.band = ctx->peer_bss_rf_band;

					trace_topology_tree_cb(ctx, ctx->root_leaf, (topology_tree_cb_func)config_state_set_handler, (void*)&sync_data);
	                msg_leng = ap_autoconfiguration_wsc_message(msg, ctx, dmac, MESSAGE_TYPE_M2);
				}
                break;
            case e_ap_autoconfiguration_renew:
                msg_leng = ap_autoconfiguration_renew_message(msg, ctx);
                break;
			case e_ap_capability_query:
				msg_leng = ap_capability_query_message(msg, ctx);
				break;
			case e_channel_preference_query:
				msg_leng = channel_preference_query_message(msg, ctx);
				break;
			case e_channel_selection_request:
				if (ctx->role == CONTROLLER) {
					find_agent_info(ctx, dmac, &agent);
					if (!agent && !ctx->send_tlv_len) {
						debug(DEBUG_ERROR, "error! no agent info exist && send_tlv_len==0\n");
						break;
					}
					msg_leng = channel_selection_request_message(msg, agent, ctx);
			}
				break;
			case e_combined_infrastructure_metrics:
				if (ctx->role == CONTROLLER)
					msg_leng = combined_infrastructure_metrics_message(msg, ctx, dmac);
				else
					reset_send_tlv(ctx);
				break;
			case e_ap_capability_report:
				msg_leng = ap_capability_report_message(msg, ctx);
				break;
			case e_cli_capability_query:
				msg_leng = cli_capability_query_message(msg, ctx);
				break;
			case e_cli_capability_report:
				msg_leng = cli_capability_report_message(msg, ctx);
				break;
			case e_channel_selection_response:
				msg_leng = channel_selection_response_message(msg, ctx);
				break;
			case e_channel_preference_report:
				msg_leng = channel_preference_report_message(msg, ctx);
                break;
			case e_operating_channel_report:
				msg_leng = operating_channel_report_message(msg, ctx);
				break;
			case e_client_steering_btm_report:
				msg_leng = client_steering_btm_report_message(msg, ctx);
				break;
			case e_steering_completed:
				msg_leng = client_steering_completed_message(msg, ctx);
				break;
			case e_multi_ap_policy_config_request:
				msg_leng = map_policy_config_request_message(msg, ctx);
				break;
			case e_client_association_control_request:
				msg_leng = client_association_control_request_message(msg, ctx);
				break;
			case e_ap_metrics_query:
				msg_leng = ap_metrics_query_message(msg, ctx);
				break;
			case e_ap_metrics_response:
				msg_leng = ap_metrics_response_message(msg, ctx);
				break;
			case e_associated_sta_link_metrics_query:
				msg_leng = associated_sta_link_metrics_query_message(msg, ctx);
				break;
			case e_associated_sta_link_metrics_response:
				msg_leng = associated_sta_link_metrics_response_message(msg, ctx);
				break;
			case e_unassociated_sta_link_metrics_query:
				msg_leng = unassociated_sta_link_metrics_query_message(msg, ctx);
				break;
			case e_unassociated_sta_link_metrics_response:
				msg_leng = unassociated_sta_link_metrics_response_message(msg, ctx);
				break;
			case e_beacon_metrics_query:
				msg_leng = beacon_metrics_query_message(msg, ctx);
				break;
			case e_beacon_metrics_response:
				msg_leng = beacon_metrics_response_message(msg, ctx);
				break;
			case e_backhaul_steering_response:
				msg_leng = backhaul_steering_response_message(msg, ctx);
				break;
			case e_higher_layer_data:
				msg_leng = high_layer_data_message(msg, ctx);
				break;
			case e_1905_ack:
				msg_leng = _1905_ack_message(msg, ctx);
				break;
			case e_dev_send_1905_request:
				msg_leng = dev_send_1905_msg(msg, ctx);
				break;
			case e_client_steering_request:
				msg_leng = client_steering_request_message(msg, ctx);
				break;
			case e_backhaul_steering_request:
				msg_leng= backhaul_steering_request_message(msg, ctx);
				break;

#ifdef MAP_R2
			/*channel scan feature*/
			case e_channel_scan_request:
				if (ctx->role == CONTROLLER)
					msg_leng = channel_scan_request_message(msg, ctx);
				break;
			case e_channel_scan_report:
				msg_leng = channel_scan_report_message(msg, ctx);
				break;

			case e_tunneled_message:
				msg_leng = tunneled_message(msg, ctx);
				break;
				
			case e_association_status_notification:
				msg_leng = assoc_status_notification_message(msg, ctx);
				break;
				
			case e_cac_request:
				msg_leng = cac_request_message(msg, ctx);
				break;
				
			case e_cac_termination:
				msg_leng = cac_terminate_message(msg, ctx);
				break;

			case e_client_disassociation_stats:
				msg_leng = client_disassciation_stats_message(msg, ctx);
				break;

			case e_failed_association_message:
				msg_leng = failed_association_message(msg, ctx);
				break;
#endif // #ifdef MAP_R2
            default:
                break;
    }

	/*move tlv_temp to buffer*/
	memcpy(tlv, get_tlv_buffer(), msg_leng);
	len = msg_leng + CMDU_HLEN + ETH_HLEN;

    msg_hdr->message_id = cpu2be16(mid);
    /* temporarily set the fragment id & last fragment ind
     	  * these two variables will be updated.
        */
    msg_hdr->fragment_id = 0;
    msg_hdr->last_fragment_indicator = 1;

	/*recheck relay indicator*/
	if (msg_hdr->relay_indicator == 0x1) {
		if (memcmp(dmac, p1905_multicast_address, ETH_ALEN)) {
			msg_hdr->relay_indicator = 0x0;
		}
	}

	/*insert message into retry queue to monitor if 1905_ack is received from the peer*/
	if (exist_in_retry_message_list(be2cpu16(msg_hdr->message_type)))
		insert_retry_message_queue(dmac, be2cpu16(msg_hdr->message_id),
			be2cpu16(msg_hdr->message_type), RETRY_CNT,
			if_index, buffer, len);

	/*exceed the max frame size, do need to fragment TX*/
    if (msg_leng > MAX_TLVS_LENGTH) {
        if (0 > cmdu_tx_fragment(ctx, buffer, len, if_index)) {
			debug(DEBUG_ERROR, "send fragment failed on %s (%s)\n", ctx->itf[if_index].if_name,
                   strerror(errno));
            return -1;
        }
    } else {
		/*send to local interface*/
		if(if_index != FIRST_VITUAL_ITF) {
        	len = send(if_index < 0 ? ctx->sock_br : ctx->sock_inf_recv_1905[if_index], buffer, len, 0);
		}
		/*send to virtual interface*/
		else if(if_index == FIRST_VITUAL_ITF)
			len = concurrent_sock_send(ctx, (char *)buffer, len);
        if (0 > len) {
			if (errno == ENETDOWN) {
				debug(DEBUG_ERROR, "send failed on %s (%s)\n",ctx->itf[if_index].if_name,
					strerror(errno));
				return 0;
			}
            return -1;
        }
		debug(DEBUG_TRACE,"msgtype=0x%04x, msg_hdr->message_id=%d interface_id=%d interface=%s\n",
			be2cpu16(msg_hdr->message_type), be2cpu16(msg_hdr->message_id), if_index,
			(if_index < 0 ? ctx->br_name : ctx->itf[if_index].if_name));
#if 0
		if (be2cpu16(msg_hdr->message_type) == CHANNLE_PREFERENCE_REPORT)
			hex_dump_all("CHANNLE_PREFERENCE_REPORT", buffer, len);
#endif
    }
    return 0;
}




/**
 *  for cmdu message relay.
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \param  buf  pointer of buffer for tx.
 * \param  len  transmit length.
 * \return error code
 */
int cmdu_bridge_relay(struct p1905_managerd_ctx *ctx,
                             unsigned char *buf, int len, int if_index)
{
    int length = 0;

	if(if_index != FIRST_VITUAL_ITF)
    {
    	if (if_index == -1)
    		length = send(ctx->sock_br, buf, len, 0);
		else
			length = send(ctx->sock_inf_recv_1905[if_index], buf, len, 0);
	}
	else
	{
		if(ctx->concurrent == 1)
		{
			length = concurrent_sock_send(ctx, (char *)buf, len);
		}
	}

    if(0 > length)
    {
		if (errno == ENETDOWN) {
			debug(DEBUG_ERROR, "send failed on %s (%s)\n", (char *)ctx->itf[if_index].if_name,
				strerror(errno));
			return 0;
		}
        return -1;
    }

    return 0;

}

/**
 *  for cmdu message relay.
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \param  buf  pointer of buffer for tx.
 * \param  len  transmit length.
 * \return error code
 */
int cmdu_bridge_relay_send(struct p1905_managerd_ctx *ctx,
            unsigned char *smac, unsigned char *buf, int len)
{
    int i = 0;

	for(i = (ctx->recent_cmdu_rx_if_number == FIRST_VITUAL_ITF || ctx->concurrent == 0) ? 1 : 0 ;
		i < ctx->itf_number; i++)
    {
        /*do not relay to original receive interface*/
        if(i != ctx->recent_cmdu_rx_if_number)
        {
            if(0 > cmdu_bridge_relay(ctx, buf, len, i))
            {
                return -1;
            }
        }
    }
    return 0;
}

/**
 *  receive cmdu message from virtual interface.
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \param  buf  pointer of buffer for rx.
 * \param  len  max receive length.
 * \return error code
 */
int cmdu_virtual_if_receive(struct p1905_managerd_ctx *ctx, int sock, unsigned char *buf, int len)
{
	struct sockaddr_un from;
	int lenfrom = sizeof(from);
	int recv_fd = 0;

    recv_fd = accept(ctx->sock_inf_recv_1905[FIRST_VITUAL_ITF], (struct sockaddr *)&from, (socklen_t *)&lenfrom);
    if(recv_fd < 0)
    {
        debug(DEBUG_ERROR, "accept fail\n");
        return -1;
    }

	len = recv(recv_fd, buf, len, 0);

    if(0 >= len)
    {
        debug(DEBUG_ERROR, "recv fail\n");
        close(recv_fd);
        return -1;
    }
	close(recv_fd);

	buf[len] = '\0';

    return len;
}

/**
 *  receive cmdu message from socket.
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \param  buf  pointer of buffer for rx.
 * \param  len  max receive length.
 * \return error code
 */
int cmdu_bridge_receive(struct p1905_managerd_ctx *ctx, int sock, unsigned char *buf, int len)
{
    len = recv(sock, buf, len, 0);
    if(0 >= len)
    {
        debug(DEBUG_WARN, "receive failed on %s (%s)", ctx->br_name,
               strerror(errno));
        return -1;
    }

	buf[len] = '\0';
    return len;
}

/**
 *  parse cmdu.
 *
 * \param  ctx  p1905_managerd_ctx context.
 * \param  buf  pointer of buffer for rx.
 * \param  len  receive length.
 * \return error code
 */
int cmdu_parse(struct p1905_managerd_ctx *ctx, unsigned char *buf, int len)
{
    struct ethhdr *eth_hdr;
    unsigned char *temp_buf;

    eth_hdr = (struct ethhdr*)buf;

	if (!(ctx->itf[ctx->recent_cmdu_rx_if_number].trx_config & RX_UNI) &&
		os_memcmp(eth_hdr->h_dest, p1905_multicast_address, ETH_ALEN)) {
			debug(DEBUG_TRACE, "drop rx unicast packet on %s\n",
				ctx->itf[ctx->recent_cmdu_rx_if_number].if_name);
		return 0;
	}

    /*examine the dst mac address to decide if we need to parse this packet*/
   if(!cmdu_dst_mac_check(ctx, eth_hdr->h_dest))
   		return 0;

	/*examine the received cmdu is from us, then drop it*/
	if(cmdu_src_mac_check(ctx, eth_hdr->h_source))
		return 0;

    /*shift to cmdu message start position*/
    temp_buf = buf;
    temp_buf += ETH_HLEN;

    if(0 > parse_cmdu_message(ctx,temp_buf,eth_hdr->h_dest,eth_hdr->h_source,len))
    {
        debug(DEBUG_ERROR, "parse_cmdu_message fail\n");
        return -1;
    }

    if(ctx->need_relay)
    {
        ctx->need_relay = 0;
		/*change to our almac, because some device may drop multicast not sending by neighbor*/
		memcpy(eth_hdr->h_source, ctx->p1905_al_mac_addr, ETH_ALEN);
        if(0 > cmdu_bridge_relay_send(ctx, eth_hdr->h_source, buf, len))
        {
            debug(DEBUG_ERROR, "cmdu_bridge_relay_send fail\n");
            return -1;
        }
    }

    return 0;
}

